/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples_PMU
  * @{
  */

/* Only one macro can be opened at a time */
#define LP_MODE_STOP1
//#define LP_MODE_STOP2
//#define LP_MODE_STANDBY


/**
  * @brief  Initializate pin of exti module.
  * @retval None
  */
void exti_pin_init(void)
{
	gpio_init_t x;

	/* Initialize GPIO_PIN_1 as input */
	x.mode  = GPIO_MODE_INPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_DOWN;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.func  = GPIO_FUNC_1;
	ald_gpio_init(PMU_EXTI_PORT, PMU_EXTI_PIN, &x);
}

/**
  * @brief  Initializate pin of led module.
  * @retval None
  */
void led_pin_init(void)
{
	gpio_init_t x;

	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_1;

	ald_gpio_init(GPIOF, GPIO_PIN_0, &x);
	ald_gpio_write_pin(GPIOF, GPIO_PIN_0, 1);

	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	exti_init_t exti;

	/* Initialize ALD */
	ald_cmu_init();
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	/* Enable external interrupt */
	ald_mcu_irq_config(EXTI1_IRQn, 0, 3, ENABLE);

	exti_pin_init();
	led_pin_init();

	ald_gpio_write_pin(GPIOF, GPIO_PIN_0, 0);
	ald_delay_ms(4000);
	ald_gpio_write_pin(GPIOF, GPIO_PIN_0, 1);

	/* Initialize external interrupt */
	exti.filter      = ENABLE;
	exti.cks         = EXTI_FILTER_CLOCK_10K;
	exti.filter_time = 10;
	ald_gpio_exti_init(PMU_EXTI_PORT, PMU_EXTI_PIN, &exti);

	/* Clear interrupt flag */
	ald_gpio_exti_clear_flag_status(PMU_EXTI_PIN);
	/* Configure interrupt */
	ald_gpio_exti_interrupt_config(PMU_EXTI_PIN, EXTI_TRIGGER_RISING_EDGE, ENABLE);

#ifdef LP_MODE_STOP1
	ald_pmu_stop1_enter();
#endif
#ifdef LP_MODE_STOP2
	ald_pmu_stop2_enter();
#endif
#ifdef LP_MODE_STANDBY
	ald_pmu_standby_enter(PMU_STANDBY_PORT_SEL_PA1, PMU_STANDBY_LEVEL_HIGH);
#endif

	while (1) {
		ald_gpio_write_pin(GPIOF, GPIO_PIN_0, 0);
		ald_delay_ms(400);
		ald_gpio_write_pin(GPIOF, GPIO_PIN_0, 1);
		ald_delay_ms(400);
	}
}

/**
  * @brief  GPIO IRQ handler
  * @retval None
  */
__isr__ void gpio_irq_handler(void)
{
	/* Check and clear interrupt flag */
	ald_gpio_exti_get_flag_status(PMU_EXTI_PIN);
	ald_gpio_exti_clear_flag_status(PMU_EXTI_PIN);

	return;
}
