/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
adc_handle_t h_adc;
pis_handle_t h_pis;
uint8_t cnt = 0;

/**
  * @brief  Adc normal convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void normal_complete_cbk(adc_handle_t *arg)
{
	++cnt;
}

/**
  * @brief  Initializate ADC.
  * @retval None.
  */
void adc_init(void)
{
	adc_nch_conf_t nm_config;

	/* clear structures */
	memset(&h_adc, 0x0, sizeof(adc_handle_t));
	memset(&nm_config, 0x0, sizeof(adc_nch_conf_t));

	/* Initialize adc */
	h_adc.perh            = ADC0;
	h_adc.init.align      = ADC_DATAALIGN_RIGHT;
	h_adc.init.scan       = DISABLE;
	h_adc.init.cont       = DISABLE;
	h_adc.init.nch_nr     = ADC_NCH_NR_1;
	h_adc.init.disc       = ADC_ALL_DISABLE;
	h_adc.init.disc_nr    = ADC_DISC_NR_1;
	h_adc.init.data_bit   = ADC_CONV_BIT_8;
	h_adc.init.div        = ADC_CKDIV_8;
	h_adc.init.nche_sel   = ADC_NCHESEL_MODE_ALL;
	h_adc.init.n_ref      = ADC_NEG_REF_VSS;
	h_adc.init.p_ref      = ADC_POS_REF_VDD;
	h_adc.normal_cplt_cbk = normal_complete_cbk;
	h_adc.insert_cplt_cbk = NULL;
	h_adc.wdg_cbk         = NULL;
	h_adc.error_cbk       = NULL;
	h_adc.ovr_cbk         = NULL;
	ald_adc_init(&h_adc); 
	
	ADC_NETS_ENABLE(&h_adc, 0x03);
	/* Initialize normal convert channel */
	nm_config.ch   = ADC_CHANNEL_15;
	nm_config.idx  = ADC_NCH_IDX_1;
	nm_config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &nm_config);
	ADC_ENABLE(&h_adc);
	ADC0->CLR = ADC_FLAG_NCH;
	ald_adc_interrupt_config(&h_adc, ADC_IT_NCH, ENABLE);
}

/**
  * @brief  Initializate toggle pin and ADC Channel15 pin.
  * @retval None.
  */
void pin_init(void)
{
	gpio_init_t x;

	memset(&x, 0, sizeof(gpio_init_t));
	/* Initialize pin */
	x.mode  = GPIO_MODE_INPUT;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_CMOS;
	x.func  = GPIO_FUNC_1;
	ald_gpio_init(GPIOB, GPIO_PIN_6, &x);
	
	/* Initialize ADC CH15 pin */
	x.mode  = GPIO_MODE_INPUT;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_CMOS;
	x.func  = GPIO_FUNC_0;
	ald_gpio_init(GPIOB, GPIO_PIN_1, &x);
}

/**
  * @brief  Initializate PIS.
  * @retval None.
  */
void pis_init(void)
{
	/* Clear pis_handle_t struct */
	memset(&h_pis, 0, sizeof(pis_handle_t));
	/* Initialize PIS */
	h_pis.perh                 = PIS;
	h_pis.init.producer_src    = PIS_GPIO_PIN6;
	h_pis.init.producer_clk    = PIS_CLK_PCLK1;
	h_pis.init.producer_edge   = PIS_EDGE_UP_DOWN;
	h_pis.init.producer_signal = PIS_OUT_LEVEL;
	h_pis.init.consumer_trig   = PIS_CH6_ADC0_NORMAL; 
	h_pis.init.consumer_clk    = PIS_CLK_PCLK2;
	ald_pis_create(&h_pis);
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	ald_mcu_irq_config(ADC0_IRQn, 1, 1, ENABLE);
	
	pin_init();
	adc_init();
	pis_init();
	
	while (1);
}
/**
  * @}
  */

/**
  * @}
  */
