/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"
#include <string.h>

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
pis_handle_t h_pis;
timer_handle_t h_tim0;
timer_handle_t h_tim1;
timer_handle_t h_tim2;

timer_oc_init_t tim0_ocinit;
timer_oc_init_t tim1_ocinit;
timer_slave_config_t slave_config;
/**
  * @brief  Initializate timer module.
  * @retval None
  */
void pis_tim_init(void)
{
	/* clear timer_handle_t structure */
	memset(&h_tim0,0,sizeof(timer_handle_t));
	memset(&h_tim1,0,sizeof(timer_handle_t));
	memset(&h_tim2,0,sizeof(timer_handle_t));
	/* clear timer_slave_config_t structure */
	memset(&slave_config,0,sizeof(timer_slave_config_t));
	
	/* Initialize TIMER2 */
	h_tim2.perh           = TIMER2;
	h_tim2.init.prescaler = 0x00;
	h_tim2.init.mode      = TIMER_CNT_MODE_UP;
	h_tim2.init.period    = 0x300;
	h_tim2.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim2.init.re_cnt    = 0;
	ald_timer_base_init(&h_tim2);

	/* Initialize TIMER0 */
	h_tim0.perh           = TIMER0;
	h_tim0.init.prescaler = 0x00;
	h_tim0.init.mode      = TIMER_CNT_MODE_UP;
	h_tim0.init.period    = 0x300;
	h_tim0.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim0.init.re_cnt    = 0;
	ald_timer_base_init(&h_tim0);
	
	slave_config.mode  = TIMER_MODE_TRIG;	/* Trigger mode */
	slave_config.input = TIMER_TS_ITR0;     /* Trigger source ITR1 */
	slave_config.polarity = TIMER_CLK_POLARITY_RISE;
	slave_config.psc      = TIMER_ETR_PSC_DIV1;
	ald_timer_slave_config_sync(&h_tim0, &slave_config);
	
	/* Common configuration for all channels */
	tim0_ocinit.oc_mode      = TIMER_OC_MODE_PWM1;
        tim0_ocinit.oc_polarity  = TIMER_OC_POLARITY_HIGH;
        tim0_ocinit.oc_fast_en   = DISABLE;
        tim0_ocinit.ocn_polarity = TIMER_OCN_POLARITY_HIGH;
        tim0_ocinit.ocn_idle     = TIMER_OCN_IDLE_RESET;
        tim0_ocinit.oc_idle      = TIMER_OC_IDLE_RESET;

        /* Set the pulse value for channel 1 */
        tim0_ocinit.pulse = 0X1F0;
	ald_timer_oc_config_channel(&h_tim0, &tim0_ocinit, TIMER_CHANNEL_1);
	
	/* Initialize TIMER1 */
	h_tim1.perh           = TIMER1;
	h_tim1.init.prescaler = 0x00;
	h_tim1.init.mode      = TIMER_CNT_MODE_UP;
	h_tim1.init.period    = 0x300;
	h_tim1.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim1.init.re_cnt    = 0;
	ald_timer_base_init(&h_tim1);

	slave_config.mode  = TIMER_MODE_TRIG;	/* Trigger mode */
	slave_config.input = TIMER_TS_ITR1;     /* Trigger source ITR1 */
	slave_config.polarity = TIMER_CLK_POLARITY_RISE;
	slave_config.psc      = TIMER_ETR_PSC_DIV1;
	ald_timer_slave_config_sync(&h_tim1, &slave_config);

	/* Common configuration for all channels */
	tim1_ocinit.oc_mode      = TIMER_OC_MODE_PWM1;
        tim1_ocinit.oc_polarity  = TIMER_OC_POLARITY_HIGH;
        tim1_ocinit.oc_fast_en   = DISABLE;
        tim1_ocinit.ocn_polarity = TIMER_OCN_POLARITY_HIGH;
        tim1_ocinit.ocn_idle     = TIMER_OCN_IDLE_RESET;
        tim1_ocinit.oc_idle      = TIMER_OC_IDLE_RESET;

        /* Set the pulse value for channel 1 */
        tim1_ocinit.pulse = 0X1F0;
	ald_timer_oc_config_channel(&h_tim1, &tim1_ocinit, TIMER_CHANNEL_1);
	
	/* Enable timer channel */
	TIMER_CCx_ENABLE(&h_tim0, 0);
	TIMER_CCx_ENABLE(&h_tim1, 0);
	
	TIMER_MOE_ENABLE(&h_tim0);
	TIMER_MOE_ENABLE(&h_tim1);
}

/**
  * @brief  Start timer2 module.
  * @retval None
  */
void pis_tim_start(void)
{
	ald_timer_base_start(&h_tim2);
}

/**
  * @brief  Stop timer2 module.
  * @retval None
  */
void pis_tim_stop(void)
{
	ald_timer_base_stop(&h_tim2);
}

/**
  * @brief  Timer0 and timer1 channel1 init.
  * @retval None
  */
void timer_pin_init(void)
{
        gpio_init_t x;

	/* Initialize tim0 ch1 pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.func = GPIO_FUNC_2;

	ald_gpio_init(GPIOE, GPIO_PIN_9, &x);
        ald_gpio_init(GPIOC, GPIO_PIN_6, &x);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	/* Clear pis_handle_t struct */
	memset(&h_pis, 0, sizeof(pis_handle_t));

	timer_pin_init();
	
	/* Initialize timer */
	pis_tim_init();

	/* Initialize PIS */
	h_pis.perh                 = PIS;
	h_pis.init.producer_src    = PIS_TIMER2_UPDATA;
	h_pis.init.producer_clk    = PIS_CLK_PCLK1;
	h_pis.init.producer_edge   = PIS_EDGE_NONE;
	h_pis.init.producer_signal = PIS_OUT_PULSE;
	h_pis.init.consumer_trig   = PIS_CH12_TIMER0_ITR0; 
	h_pis.init.consumer_clk    = PIS_CLK_PCLK1;
	ald_pis_create(&h_pis);
	
	h_pis.init.consumer_trig   = PIS_CH13_TIMER1_ITR1; 
	ald_pis_create(&h_pis);
	
	pis_tim_start();
	
	while (1);
}
/**
  * @}
  */

/**
  * @}
  */
