/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"
#include <string.h>
  
/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

pis_handle_t h_pis;
timer_handle_t h_tim;
uart_handle_t h_uart;

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void uart_pin_init(void)
{
	gpio_init_t x;
	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_10, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_11, &x);

	return;
}

/**
  * @brief  Initializate timer0 module.
  * @retval None
  */
void pis_tim_init(void)
{
	/* clear timer_handle_t structure */
	memset(&h_tim, 0, sizeof(timer_handle_t));

	/* Initialize TIMER0 */
	h_tim.perh           = TIMER0;
	h_tim.init.prescaler = 0x10;
	h_tim.init.mode      = TIMER_CNT_MODE_UP;
	h_tim.init.period    = 0x300;
	h_tim.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim.init.re_cnt    = 0;
	ald_timer_base_init(&h_tim);
}

/**
  * @brief  Initializate Uart0 module.
  * @retval None
  */
void pis_uart_init(void)
{
	/* clear timer_handle_t structure */
	memset(&h_uart, 0, sizeof(uart_handle_t));

	/* Initialize TIMER0 */
	h_uart.perh             = UART0;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	ald_uart_init(&h_uart);
}

/**
  * @brief  Start timer0 module.
  * @retval None
  */
void pis_tim_start(void)
{
	ald_timer_base_start(&h_tim);
}

/**
  * @brief  Stop timer0 module.
  * @retval None
  */
void pis_tim_stop(void)
{
	ald_timer_base_stop(&h_tim);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);

	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	/* Clear pis_handle_t struct */
	memset(&h_pis, 0, sizeof(pis_handle_t));

	/* Initialize pin */
	uart_pin_init();
	/* Initialize timer0 */
	pis_tim_init();
	/* Initialize uart0 */
	pis_uart_init();

	/* Initialize PIS */
	h_pis.perh                 = PIS;
	h_pis.init.producer_src    = PIS_TIMER0_UPDATA;
	h_pis.init.producer_clk    = PIS_CLK_PCLK1;
	h_pis.init.producer_edge   = PIS_EDGE_NONE;
	h_pis.init.producer_signal = PIS_OUT_LEVEL;
	h_pis.init.consumer_trig   = PIS_CH9_UART0_RXD; 
	h_pis.init.consumer_clk    = PIS_CLK_PCLK1;
	h_pis.init.input_chan      = PIS_PORT_INPUT;
	ald_pis_create(&h_pis);

	pis_tim_start();

	while (1) {
		ald_delay_ms(1000);
	}
}
/**
  * @}
  */

/**
  * @}
  */
