/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
bsp_key_env_t bsp_key_env;
uart_handle_t h_uart;
char uart_buf[64];
uint32_t adc_value;
uint32_t __value = 0;
adc_handle_t h_adc;
adc_nch_conf_t nch_config;

void bsp_light_init(void);
void bsp_light_ctrl(bsp_light_idx_t idx, bsp_light_ctrl_t ctrl);
void bsp_key_init(void);
void bsp_key_scan(void);
void bsp_key_bond_func(bsp_key_event_t event, bsp_key_cbk_t cbk);
void bsp_key_unbond_func(bsp_key_event_t event);
void bsp_adc_init(void);
uint32_t bsp_adc_value_get(void);
void printf_e(const char *fmt, ...);
void uart_stdio_init(void);

/**
  * @brief  Hnadle key event.
  * @param  event: Key event.
  * @retval None
  */
void key_handle(bsp_key_event_t event)
{
	switch ((uint32_t)event) {
	case BSP_KEY1_PRESS:
		printf_e("\rUP Press\n\r");
		bsp_light_ctrl(BSP_LIGHT_1, BSP_LIGHT_TOGGLE);
		break;
	case BSP_KEY1_RELEASE:
		printf_e("\rUP Release\n\r");
		bsp_light_ctrl(BSP_LIGHT_1, BSP_LIGHT_TOGGLE);
		break;
	case BSP_KEY2_PRESS:
		printf_e("\rDOWN Press\n\r");
		bsp_light_ctrl(BSP_LIGHT_2, BSP_LIGHT_TOGGLE);
		break;
	case BSP_KEY2_RELEASE:
		printf_e("\rDOWN Release\n\r");
		bsp_light_ctrl(BSP_LIGHT_2, BSP_LIGHT_TOGGLE);
		break;
	case BSP_KEY3_PRESS:
		printf_e("\rMID Press\n\r");
		bsp_light_ctrl(BSP_LIGHT_2, BSP_LIGHT_TOGGLE);
		bsp_light_ctrl(BSP_LIGHT_1, BSP_LIGHT_TOGGLE);
		break;
	case BSP_KEY3_RELEASE:
		printf_e("\rMID Release\n\r");
		bsp_light_ctrl(BSP_LIGHT_2, BSP_LIGHT_TOGGLE);
		bsp_light_ctrl(BSP_LIGHT_1, BSP_LIGHT_TOGGLE);
		break;
	case BSP_KEY4_PRESS:
		printf_e("\rLEFT Press\n\r");
		bsp_light_ctrl(BSP_LIGHT_1, BSP_LIGHT_TOGGLE);
		break;
	case BSP_KEY4_RELEASE:
		printf_e("\rLEFT Release\n\r");
		bsp_light_ctrl(BSP_LIGHT_1, BSP_LIGHT_TOGGLE);
		break;
	case BSP_KEY5_PRESS:
		printf_e("\rRIGHT Press\n\r");
		bsp_light_ctrl(BSP_LIGHT_2, BSP_LIGHT_TOGGLE);
		break;
	case BSP_KEY5_RELEASE:
		printf_e("\rRIGHT Release\n\r");
		bsp_light_ctrl(BSP_LIGHT_2, BSP_LIGHT_TOGGLE);
		break;

	default:
		break;
	}

	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	int diff;

	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
	/* Enable peripheral */
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_UART1, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_ADC0, ENABLE);

	/* Initialize peripheral */
	bsp_light_init();
	uart_stdio_init();
	bsp_adc_init();
	bsp_key_init();
	bsp_key_bond_func(BSP_KEY_ALL, key_handle);
	printf_e("\rSystem startup...\n\r");

	while (1) {
		adc_value = bsp_adc_value_get();
		diff = adc_value > __value ? adc_value - __value : __value - adc_value;

		if (diff > 7) {
			__value = adc_value;
			printf_e("\rADC Value: %dmV\n\r", adc_value);
		}
	}
}


/**
  * @brief  Initialize light.
  * @retval None
  */
void bsp_light_init(void)
{
	gpio_init_t x;

	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_1;

	ald_gpio_init(BSP_LED1_PORT, BSP_LED1_PIN, &x);
	ald_gpio_init(BSP_LED2_PORT, BSP_LED2_PIN, &x);
	bsp_light_ctrl(BSP_LIGHT_1, BSP_LIGHT_OFF);
	bsp_light_ctrl(BSP_LIGHT_2, BSP_LIGHT_OFF);

	return;
}

/**
  * @brief  Control light.
  * @param  idx: Index of the light.
  * @param  ctrl: Type: ON/OFF/TOGGLE.
  * @retval None
  */
void bsp_light_ctrl(bsp_light_idx_t idx, bsp_light_ctrl_t ctrl)
{
	if (idx == BSP_LIGHT_1) {
		switch ((uint32_t)ctrl) {
		case BSP_LIGHT_ON:
			ald_gpio_write_pin(BSP_LED1_PORT, BSP_LED1_PIN, 0);
			break;
		case BSP_LIGHT_OFF:
			ald_gpio_write_pin(BSP_LED1_PORT, BSP_LED1_PIN, 1);
			break;
		case BSP_LIGHT_TOGGLE:
			ald_gpio_toggle_pin(BSP_LED1_PORT, BSP_LED1_PIN);
			break;
		default:
			break;
		}
	}
	else if (idx == BSP_LIGHT_2) {
		switch ((uint32_t)ctrl) {
		case BSP_LIGHT_ON:
			ald_gpio_write_pin(BSP_LED2_PORT, BSP_LED2_PIN, 0);
			break;
		case BSP_LIGHT_OFF:
			ald_gpio_write_pin(BSP_LED2_PORT, BSP_LED2_PIN, 1);
			break;
		case BSP_LIGHT_TOGGLE:
			ald_gpio_toggle_pin(BSP_LED2_PORT, BSP_LED2_PIN);
			break;
		default:
			break;
		}
	}
	else {
		return;
	}

	return;	
}

/**
  * @brief  Initialize key's pin.
  * @retval None
  */
void bsp_key_pin_init(void)
{
	gpio_init_t x;

	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_1;

	ald_gpio_init(BSP_KEY1_PORT, BSP_KEY1_PIN, &x);
	ald_gpio_init(BSP_KEY2_PORT, BSP_KEY2_PIN, &x);
	ald_gpio_init(BSP_KEY3_PORT, BSP_KEY3_PIN, &x);
	ald_gpio_init(BSP_KEY4_PORT, BSP_KEY4_PIN, &x);
	ald_gpio_init(BSP_KEY5_PORT, BSP_KEY5_PIN, &x);

	return;
}

/**
  * @brief  Read key1's value
  * @retval The status:
  *           0 - No press
  *           1 - Press
  */
uint8_t bsp_key_read_key1(void)
{
	return !ald_gpio_read_pin(BSP_KEY1_PORT, BSP_KEY1_PIN);
}

/**
  * @brief  Read key2's value
  * @retval The status:
  *           0 - No press
  *           1 - Press
  */
uint8_t bsp_key_read_key2(void)
{
	return !ald_gpio_read_pin(BSP_KEY2_PORT, BSP_KEY2_PIN);
}

/**
  * @brief  Read key3's value
  * @retval The status:
  *           0 - No press
  *           1 - Press
  */
uint8_t bsp_key_read_key3(void)
{
	return !ald_gpio_read_pin(BSP_KEY3_PORT, BSP_KEY3_PIN);
}

/**
  * @brief  Read key4's value
  * @retval The status:
  *           0 - No press
  *           1 - Press
  */
uint8_t bsp_key_read_key4(void)
{
	return !ald_gpio_read_pin(BSP_KEY4_PORT, BSP_KEY4_PIN);
}

/**
  * @brief  Read key4's value
  * @retval The status:
  *           0 - No press
  *           1 - Press
  */
uint8_t bsp_key_read_key5(void)
{
	return !ald_gpio_read_pin(BSP_KEY5_PORT, BSP_KEY5_PIN);
}


/**
  * @brief  Initialization key module.
  * @retval None
  */
void bsp_key_init(void)
{
	memset(&bsp_key_env, 0x0, sizeof(bsp_key_env_t));
	bsp_key_pin_init();

	bsp_key_env.read_cbk[0] = bsp_key_read_key1;
	bsp_key_env.read_cbk[1] = bsp_key_read_key2;
	bsp_key_env.read_cbk[2] = bsp_key_read_key3;
	bsp_key_env.read_cbk[3] = bsp_key_read_key4;
	bsp_key_env.read_cbk[4] = bsp_key_read_key5;

	return;
}

/**
  * @brief  Scan key.
  * @retval None
  */
void bsp_key_scan(void)
{
	int32_t i;

	for (i = 0; i < BSP_KEY_MAX; ++i) {
		if (!(bsp_key_env.valid & (1U << i)))
			continue;

		if ((bsp_key_env.read_cbk[i]() == 1) && (bsp_key_env.state[i].press == 0)) {
			if (++bsp_key_env.state[i].hold >= 3) {
				bsp_key_env.state[i].hold  = 0;
				bsp_key_env.state[i].press = 1;

				if (bsp_key_env.cbk[i << 1])
					bsp_key_env.cbk[i << 1]((bsp_key_event_t)(i << 1));
			}
		}
		else if ((bsp_key_env.read_cbk[i]() == 0) && (bsp_key_env.state[i].press == 1)) {
			if (++bsp_key_env.state[i].hold >= 3) {
				bsp_key_env.state[i].hold  = 0;
				bsp_key_env.state[i].press = 0;

				if (bsp_key_env.cbk[(i << 1) + 1])
					bsp_key_env.cbk[(i << 1) + 1]((bsp_key_event_t)((i << 1) + 1));
			}
		}
	}
}

/**
  * @brief  This function invoked by Systick ISR each 1ms.
  * @retval None
  */
void ald_systick_irq_cbk(void)
{
	if (((ald_get_tick()) & 0x7) == 0)
		bsp_key_scan();
}

/**
  * @brief  Bond callback function.
  * @param  event: Type of event.
  * @param  cbk: Callback function.
  * @retval None
  */
void bsp_key_bond_func(bsp_key_event_t event, bsp_key_cbk_t cbk)
{
	int32_t i;

	if (event == BSP_KEY_ALL) {
		for (i = 0; i < BSP_KEY_MAX; ++i) {
			bsp_key_env.valid            |= (1U << i);
			bsp_key_env.cbk[i << 1]       = cbk;
			bsp_key_env.cbk[(i << 1) + 1] = cbk;
		}
		return;
	}

	bsp_key_env.valid     |= 1 << (event >> 1);
	bsp_key_env.cbk[event] = cbk;
	return;
}

/**
  * @brief  Unbond callback function.
  * @param  event: Type of event.
  * @retval None
  */
void bsp_key_unbond_func(bsp_key_event_t event)
{
	int32_t i;

	if (event == BSP_KEY_ALL) {
		for (i = 0; i < BSP_KEY_MAX; ++i) {
			bsp_key_env.valid            &= ~(1U << i);
			bsp_key_env.cbk[i << 1]       = NULL;
			bsp_key_env.cbk[(i << 1) + 1] = NULL;
		}
		return;
	}

	bsp_key_env.cbk[event] = NULL;

	if ((bsp_key_env.cbk[(event >> 1) << 1] == NULL)
			&& (bsp_key_env.cbk[((event >> 1) << 1) + 1] == NULL))
		bsp_key_env.valid  &= ~(1 << (event >> 1));

	return;
}

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsnprintf(uart_buf, 64, fmt, args);
	va_end(args);

	ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);

	return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
	gpio_init_t x;

	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_UART1, ENABLE);

	/* Initialize tx pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_0, &x);

	/* Initialize rx pin */
	x.mode  = GPIO_MODE_INPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_1, &x);

	/* Initialize uart */
	h_uart.perh             = UART1;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);

	return;
}

/**
  * @brief  Initializate pin of adc module.
  * @retval None.
  */
static void adc_pin_init(void)
{
	gpio_init_t x;

	/* Initialize ADC pin */
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(BSP_ADC_PORT, BSP_ADC_PIN, &x);

	return;
}

/**
  * @brief  Initializate adc module.
  * @retval None.
  */
void bsp_adc_init(void)
{
	adc_pin_init();

	memset(&h_adc, 0, sizeof(adc_handle_t));
	memset(&nch_config, 0, sizeof(adc_nch_conf_t));

	/* Initialize adc */
	h_adc.perh            = ADC0;
	h_adc.init.align      = ADC_DATAALIGN_RIGHT;
	h_adc.init.scan       = DISABLE;
	h_adc.init.cont       = DISABLE;
	h_adc.init.disc       = ADC_ALL_DISABLE;
	h_adc.init.disc_nr    = ADC_DISC_NR_1;
	h_adc.init.data_bit   = ADC_CONV_BIT_12;
	h_adc.init.div        = ADC_CKDIV_32;
	h_adc.init.nche_sel   = ADC_NCHESEL_MODE_ONE;
	h_adc.init.nch_nr     = ADC_NCH_NR_1;
	h_adc.init.n_ref      = ADC_NEG_REF_VSS;
	h_adc.init.p_ref      = ADC_POS_REF_2V;
	h_adc.normal_cplt_cbk = NULL;
	h_adc.insert_cplt_cbk = NULL;
	h_adc.wdg_cbk         = NULL;
	h_adc.error_cbk       = NULL;
	h_adc.ovr_cbk         = NULL;
	ald_adc_init(&h_adc); 

	/* Initialize nromal convert channel */
	nch_config.ch   = ADC_CHANNEL_9;
	nch_config.idx  = ADC_NCH_IDX_1;
	nch_config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &nch_config);

	return;
}


/**
  * @brief  Get the voltage in ms in polling mode.
  * @retval voltage in ms.
  */
uint32_t bsp_adc_value_get(void)
{
	uint32_t sum = 0;
	uint32_t value;
	int i, nr = 16;

	for (i = 0; i < nr; ++i) {
		ald_adc_normal_start(&h_adc);
		if (ald_adc_normal_poll_for_conversion(&h_adc, 5000) == OK)
			value = ald_adc_normal_get_value(&h_adc);

		sum += value;
	}

	return (((sum >> 4) * 2000) >> 12);
}

/**
  *@}
  */
/**
  *@}
  */
