/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

#define BUFFER_SIZE		0x0100
#define WRITE_READ_ADDR		0x0800
#define SRAM_MEMORY_WIDTH	EBI_NORSRAM_MEM_BUS_WIDTH_16
#define SRAM_BANK_ADDR		0x68000000

sram_handle_t h_sram;
ald_ebi_nor_sram_timing_t timing;
uint16_t tx_buf[BUFFER_SIZE];
uint16_t rx_buf[BUFFER_SIZE];
uint32_t __idx = 0;

static void sram_pin_init(void);

/**
  * @brief  Fills buffer with user predefined data.
  * @param  buf: pointer on the buffer to fill
  * @param  size: size of the buffer to fill
  * @param  offset: first value to fill on the buffer
  * @retval None
  */
static void fill_buffer(uint16_t *buf, uint32_t size, uint16_t offset)
{
	uint16_t i = 0;

	for (i = 0; i < size; ++i)
		buf[i] = i + offset;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	uint32_t i = 0;
	/* Initialize ALD */
	ald_cmu_init();
	
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_96M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 96000000);
	
	for(i = 0;i < 0xfff;++i);
	
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	
	sram_pin_init();

	h_sram.instance       = EBI_NOR_SRAM_DEVICE;
	h_sram.ext            = EBI_NOR_SRAM_EXTENDED_DEVICE;
	timing.addr_setup     = 8;
	timing.addr_hold      = 4;
	timing.data_setup     = 7;
	timing.bus_dur        = 1;
	timing.div            = 0;
	timing.latency        = 4;
	timing.mode           = EBI_ACCESS_MODE_A;
	h_sram.init.bank      = EBI_NORSRAM_BANK3;
	h_sram.init.mux       = EBI_DATA_ADDRESS_MUX_DISABLE;
	h_sram.init.type      = EBI_MEMORY_TYPE_SRAM;
	h_sram.init.width     = SRAM_MEMORY_WIDTH;
	h_sram.init.acc_mode  = EBI_BURST_ACCESS_MODE_DISABLE;
	h_sram.init.polarity  = EBI_WAIT_SIGNAL_POLARITY_LOW;
	h_sram.init.wrap_mode = EBI_WRAP_MODE_DISABLE;
	h_sram.init.active    = EBI_WAIT_TIMING_BEFORE_WS;
	h_sram.init.write     = EBI_WRITE_OPERATION_ENABLE;
	h_sram.init.signal    = EBI_WAIT_SIGNAL_DISABLE;
	h_sram.init.ext_mode  = EBI_EXTENDED_MODE_DISABLE;
	h_sram.init.wait      = EBI_ASYNCHRONOUS_WAIT_DISABLE;
	h_sram.init.burst     = EBI_WRITE_BURST_DISABLE;

	ald_sram_init(&h_sram, &timing, &timing);
	fill_buffer(tx_buf, BUFFER_SIZE, 0x5555);

	/* Write data to the SRAM memory */
	for (__idx = 0; __idx < BUFFER_SIZE; __idx++) {
		*(__IO uint16_t *)(SRAM_BANK_ADDR + WRITE_READ_ADDR + 2 * __idx) = tx_buf[__idx];
	}

	/* Read back data from the SRAM memory */
	for (__idx = 0; __idx < BUFFER_SIZE; __idx++) {
		rx_buf[__idx] = *(__IO uint16_t *)(SRAM_BANK_ADDR + WRITE_READ_ADDR + 2 * __idx);
	}

	while (1) {
		ald_delay_ms(1000);
	}
}

/**
  * @brief  Sram pin init.
  * @param  None
  * @retval None
  */
void sram_pin_init(void)
{
	uint8_t i = 0;
	gpio_init_t x;

	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.podrv = GPIO_OUT_DRIVE_0_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_7;

	/* Address pin init */
	for (i = 0;i < EBI_SRAM_ADDRESS_PINn; ++i)
		ald_gpio_init(EBI_SRAM_ADD_PORT[i], EBI_SRAM_ADD_PIN[i], &x);

	/* Data pin init */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.func = GPIO_FUNC_7;
	for(i = 0;i < EBI_SRAM_DATA_PINn; ++i) {
		ald_gpio_init(EBI_SRAM_DATA_PORT[i], EBI_SRAM_DATA_PIN[i], &x);
	}

	/* Control pin init */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.podrv = GPIO_OUT_DRIVE_0_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_7;
	/* Output enable pin init EBI_NOE */
	ald_gpio_init(EBI_NOE_PORT, EBI_NOE_PIN, &x);
	/* Write enable pin init EBI_NWE*/
	ald_gpio_init(EBI_NWE_PORT, EBI_NWE_PIN, &x);
	/* Chip select pin init EBI_NE1 */
	ald_gpio_init(EBI_NE3_PORT, EBI_NE3_PIN, &x);
	/* Low byte select EBI_NBL0 */
	ald_gpio_init(EBI_NBL0_PORT, EBI_NBL0_PIN, &x);
	/* High byte selec EBI_NBL1 */
	ald_gpio_init(EBI_NBL1_PORT, EBI_NBL1_PIN, &x);
}
/**
  * @}
  */
/**
  * @}
  */
