/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

#define WRITE_READ_ADDRESS	0x8000
#define NAND_PAGE_SIZE          0x0800	/* 2048 bytes per page w/o Spare Area */
#define NAND_BLOCK_SIZE         0x0040	/* 64x2048 bytes pages per block */
#define NAND_PLANE_SIZE         0x0800	/* 2048 Block per plane */
#define NAND_SPARE_AREA_SIZE    0x0040	/* last 16 bytes as spare area */
#define NAND_MAX_PLANE          0x1000	/* 2 planes of 1024 block */
#define NB_PAGE                 2
#define BUFFER_SIZE             0X400	/* (NAND_PAGE_SIZE * NB_PAGE) */

nand_handle_t h_nand;
ald_ebi_nand_timing_t timing;
uint8_t tx_buf[BUFFER_SIZE];
uint8_t rx_buf[BUFFER_SIZE];
static nand_id_t __id;
static nand_address_t __addr;

static void nand_pin_init(void);

/**
  * @brief  Fills buffer with user predefined data.
  * @param  buf: pointer on the buffer to fill
  * @param  size: size of the buffer to fill
  * @param  offset: first value to fill on the buffer
  * @retval None
  */
static void fill_buffer(uint8_t *buf, uint32_t size, uint32_t offset)
{
	uint16_t i = 0;

	for (i = 0; i < size; ++i)
		buf[i] = i + offset;
}

/**
  * @brief  Translate logical address into a phy one.
  * @param  addr: Address
  * @param  n_addr: NAND addr.
  * @retval Status
  */
static void nand_addr_get(uint32_t addr, nand_address_t *n_addr)
{
	n_addr->page  = (addr % (NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE))) / (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE);
	n_addr->block = (addr % (NAND_PLANE_SIZE * NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE))) / (NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE));
	n_addr->plane = addr / (NAND_PLANE_SIZE * NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE));
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_clock_config(CMU_CLOCK_HOSC, 12000000);

	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_EBI, ENABLE);

	nand_pin_init();
	
	h_nand.instance	= EBI_NAND_DEVICE;

	timing.time              = 0x05;		
	timing.wait_time         = 0xF0;
	timing.hold_time         = 0x05;		
	timing.hiz_time          = 0x08;
	h_nand.init.bank         = EBI_NAND_BANK3;
	h_nand.init.wait         = EBI_NAND_WAIT_FEATURE_ENABLE;
	h_nand.init.width        = EBI_NAND_MEM_BUS_WIDTH_8;
	h_nand.init.ecc	         = EBI_NAND_ECC_ENABLE;
	h_nand.init.size         = EBI_NAND_ECC_PAGE_SIZE_2048BYTE;
	h_nand.init.cle_time     = 0x0;
	h_nand.init.ale_time     = 0x0;
	h_nand.config.block_nbr  = NAND_MAX_PLANE;
	h_nand.config.block_size = NAND_BLOCK_SIZE;
	h_nand.config.plane_size = NAND_PLANE_SIZE;
	h_nand.config.page_size  = NAND_PAGE_SIZE;
	h_nand.config.spare_size = NAND_SPARE_AREA_SIZE;

	/* NAND Flash initialization */
	ald_nand_init(&h_nand, &timing, &timing);
	ald_nand_reset(&h_nand);

	ald_nand_read_id(&h_nand, &__id);
	nand_addr_get(WRITE_READ_ADDRESS, &__addr);

	if (ald_nand_erase_block(&h_nand, &__addr) == OK) {
	}

	fill_buffer(tx_buf, BUFFER_SIZE, 0x0);
	if (ald_nand_write_page_8b(&h_nand, &__addr, tx_buf, NB_PAGE) != OK) {
		ALD_PANIC(); 
	}

	if (ald_nand_read_page_8b(&h_nand, &__addr, rx_buf, NB_PAGE) != OK) {
		ALD_PANIC(); 
	}

	while (1) {
		ald_delay_ms(1000);
	}
}

/**
  * @brief  Nand pin init.
  * @param  None
  * @retval None
  */
void nand_pin_init(void)
{
	uint8_t i;
	gpio_init_t x;

	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.podrv = GPIO_OUT_DRIVE_0_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_7;

	/* Data pin init */
	for (i = 0;i < EBI_NAND_DATA_PINn; ++i) {
		ald_gpio_init(EBI_DATA_PORT[i], EBI_DATA_PIN[i], &x);
	}

	/* Control pin init */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.podrv = GPIO_OUT_DRIVE_0_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_7;

	/* Output enable pin init */
	ald_gpio_init(EBI_NOE_PORT, EBI_NOE_PIN, &x);
	/* Write enable init EBI_NWE*/
	ald_gpio_init(EBI_NWE_PORT, EBI_NWE_PIN, &x);
	/* Chip select pin init EBI_NCE3 */
	ald_gpio_init(EBI_NCE3_PORT, EBI_NCE3_PIN, &x);
	/* EBI_CLE init */
	ald_gpio_init(EBI_CLE_PORT, EBI_CLE_PIN, &x);
	/* Address Latch pin init EBI_ALE */
	ald_gpio_init(EBI_ALE_PORT, EBI_ALE_PIN, &x);
	/* Ready/Busy pin init */
	x.mode = GPIO_MODE_INPUT;
	x.func = GPIO_FUNC_7;
	ald_gpio_init(EBI_NWAIT_PORT, EBI_NWAIT_PIN, &x);
}
/**
  * @}
  */
/**
  * @}
  */
