/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

#define WRITE_READ_ADDRESS  0x8000
#define NOR_BANK_ADDR       0x60000000
#define NOR_TIMEOUT_VALUE   0xFFFF
#define BUFFER_SIZE         0x400

nor_handle_t h_nor;
ald_ebi_nor_sram_timing_t nor_timing;
uint16_t tx_buf[BUFFER_SIZE];
uint16_t rx_buf[BUFFER_SIZE];

static void bsp_nor_pin_init(void);
static void bsp_reset_nor_flash(void);
/**
  * @brief  Fills buffer with user predefined data.
  * @param  buf: pointer on the buffer to fill
  * @param  size: size of the buffer to fill
  * @param  offset: first value to fill on the buffer
  * @retval None
  */
static void fill_buffer(uint16_t *buf, uint32_t size, uint16_t offset)
{
	uint16_t i = 0;

	for (i = 0; i < size; ++i)
		buf[i] = i + offset;
}

/* NOR IDs structure */
static nor_id_t __id;
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	uint16_t *data = NULL;
	uint32_t index = 0;
	uint32_t addr = 0;

	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_clock_config(CMU_CLOCK_HOSC, 12000000);

	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_EBI, ENABLE);

	bsp_nor_pin_init();
	bsp_reset_nor_flash();

	h_nor.instance         = EBI_NOR_SRAM_DEVICE;
	h_nor.ext              = EBI_NOR_SRAM_EXTENDED_DEVICE;
	nor_timing.addr_setup  = 0X0F;		
	nor_timing.addr_hold   = 0x01;
	nor_timing.data_setup  = 0X10;		
	nor_timing.bus_dur     = 0;
	nor_timing.div         = 0x01;
	nor_timing.latency     = 0x01;
	nor_timing.mode        = EBI_ACCESS_MODE_B;
	h_nor.init.bank        = EBI_NORSRAM_BANK1;
	h_nor.init.mux         = EBI_DATA_ADDRESS_MUX_DISABLE;
	h_nor.init.type        = EBI_MEMORY_TYPE_NOR;
	h_nor.init.width       = EBI_NORSRAM_MEM_BUS_WIDTH_16;
	h_nor.init.acc_mode    = EBI_BURST_ACCESS_MODE_DISABLE;
	h_nor.init.polarity    = EBI_WAIT_SIGNAL_POLARITY_LOW;
	h_nor.init.wrap_mode   = EBI_WRAP_MODE_DISABLE;
	h_nor.init.active      = EBI_WAIT_TIMING_BEFORE_WS;
	h_nor.init.write       = EBI_WRITE_OPERATION_ENABLE;
	h_nor.init.signal      = EBI_WAIT_SIGNAL_DISABLE;
	h_nor.init.ext_mode    = EBI_EXTENDED_MODE_DISABLE;
	h_nor.init.wait        = EBI_ASYNCHRONOUS_WAIT_DISABLE;
	h_nor.init.burst       = EBI_WRITE_BURST_DISABLE;

	ald_nor_init(&h_nor, &nor_timing, &nor_timing);
	ald_nor_return_readmode(&h_nor);

	//Read NOR Flash ID
	ald_nor_read_id(&h_nor, &__id);
	ald_nor_return_readmode(&h_nor);

	//sector erase
	ald_nor_erase_block(&h_nor, WRITE_READ_ADDRESS, NOR_BANK_ADDR);

	if (ald_nor_get_status(&h_nor, NOR_BANK_ADDR, NOR_TIMEOUT_VALUE) == ALD_NOR_STATUS_SUCCESS) {
	
	}

	fill_buffer(tx_buf, BUFFER_SIZE, 0xC20F);

	data = tx_buf;
	index = BUFFER_SIZE;
	addr = NOR_BANK_ADDR + WRITE_READ_ADDRESS;

	while (index > 0) {
		/* Write data to NOR */
		ald_nor_program(&h_nor, (uint32_t *)addr, data);

		/* Read NOR device status */
		if(ald_nor_get_status(&h_nor, NOR_BANK_ADDR, NOR_TIMEOUT_VALUE) != ALD_NOR_STATUS_SUCCESS) {
			ALD_PANIC();
		}

		/* Update the counters */
		index--;
		addr += 2;
		data++; 
	}

	/* Read back data from the NOR memory */
	if (ald_nor_read_buffer(&h_nor, NOR_BANK_ADDR + WRITE_READ_ADDRESS, rx_buf, BUFFER_SIZE) != OK) {
		ALD_PANIC();
	}

	while (1) {
		ald_delay_ms(1000);
	}
}

/**
  * @brief  Initialize the nor pin.
  * @retval None.
  */
void bsp_nor_pin_init(void)
{
	uint8_t i = 0;
	gpio_init_t x;
	
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.podrv = GPIO_OUT_DRIVE_0_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_7;
		
	for(i = 0;i < EBI_ADDRESS_PINn; ++i) 
		ald_gpio_init(EBI_ADD_PORT[i], EBI_ADD_PIN[i], &x);
	
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.func = GPIO_FUNC_7;
	
	for(i = 0;i < EBI_DATA_PINn; ++i)
		ald_gpio_init(EBI_DATA_PORT[i], EBI_DATA_PIN[i], &x);
	
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.podrv = GPIO_OUT_DRIVE_0_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_7;
	
	ald_gpio_init(EBI_NOE_PORT, EBI_NOE_PIN, &x);
	ald_gpio_init(EBI_NWE_PORT, EBI_NWE_PIN, &x);
	ald_gpio_init(EBI_NE1_PORT, EBI_NE1_PIN, &x);
	x.func = GPIO_FUNC_1;
	ald_gpio_init(NOR_RESET_PORT, NOR_RESET_PIN, &x);
	x.mode = GPIO_MODE_INPUT;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(EBI_NWAIT_PORT, EBI_NWAIT_PIN, &x);
}

/**
  * @brief  Initialize the nor pin.
  * @retval None.
  */
void bsp_reset_nor_flash(void)
{
	uint32_t i = 0;
	ald_gpio_write_pin(NOR_RESET_PORT, NOR_RESET_PIN, 0);
	for ( i = 0xfffff; i > 0; --i);
	ald_gpio_write_pin(NOR_RESET_PORT, NOR_RESET_PIN, 1);
	for ( i = 0xfffff; i > 0; --i);
}

/**
  * @}
  */
/**
  * @}
  */
