/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

crypt_handle_t h_crypt;
uint8_t count = 0;
uint8_t error = 0;

/**
  * @brief  crypt completed callback.
  * @param  arg: Pointer to crypt_handle_t structure.
  * @retval None.
  */
void crypt_complete(crypt_handle_t *arg)
{
	ald_crypt_clear_flag_status(arg, CRYPT_FLAG_DONE);
	++count;
	return;
}

/**
  * @brief  crypt error callback.
  * @param  arg: Pointer to crypt_handle_t structure.
  * @retval None.
  */
void error_complete(crypt_handle_t *arg)
{
	++error;
	return;
}

/* des ecb key = 1 keys
 * input 	32 43 f6 a8 88 5a 30 8d
 * cipher key	2b 7e 15 16 28 ae d2 a6
 * refer result	61 cb 4a 69 bf cc 9b c4
 */
void test_des_ecb_1key()
{
	uint32_t plain_buffer[2] = {	0x3243F6A8, 0x885A308D};
	uint32_t result_buffer[2] = {0};
	uint32_t expect_buffer[2] = {	0x61cb4a69,0xbfcc9bc4};
	uint32_t key_buf[4] = {0x2b7e1516, 0x28aed2a6};

	/* Clear lpuart_handle_t structure */
	memset(&h_crypt, 0x0, sizeof(crypt_handle_t));

	/* Initialize crypt */
	h_crypt.perh           = CRYPT;
	h_crypt.init.mode      = CRYPT_MODE_ECB;//CRYPT_MODE_CBC
	h_crypt.init.key       = CRYPT_DES_KEYS_1;
	h_crypt.init.type      = DATA_32_BIT;
	h_crypt.crypt_cplt_cbk = crypt_complete;
	h_crypt.err_cplt_cbk   = error_complete;
	ald_crypt_init(&h_crypt);

	/* Start AES encrypt,right result_buffer equal to expect_buffer */
	ald_crypt_write_key(&h_crypt,key_buf,KEY_2_LEN);
 	ald_crypt_encrypt(&h_crypt,(uint8_t*)plain_buffer, (uint8_t*)result_buffer, sizeof(plain_buffer));

	/* Start AES decrypt,result_buffer equal to expect_buffer */	
	ald_crypt_write_key(&h_crypt,key_buf,KEY_2_LEN);
 	ald_crypt_decrypt(&h_crypt,(uint8_t*)expect_buffer, (uint8_t*)result_buffer, sizeof(expect_buffer));	
}

/* des ecb key = 2 keys
 * input 	32 43 f6 a8 88 5a 30 8d
 * cipher key	2b 7e 15 16 28 ae d2 a6 ab f7 15 88 09 cf 4f 3c
 * refer result	xx
 */
void test_des_ecb_2key()
{
	uint32_t plain_buffer[2] = {	0x3243F6A8, 0x885A308D};
	uint32_t result_buffer[2] = {0};
	uint32_t expect_buffer[2] = {	0x1f2bca17,0x7bc9d663};
	uint32_t key_buf[4]   = {0x2b7e1516, 0x28aed2a6, 0xabf71588, 0x09cf3c2b};

	/* Clear lpuart_handle_t structure */
	memset(&h_crypt, 0x0, sizeof(crypt_handle_t));

	/* Initialize crypt */
	h_crypt.perh           = CRYPT;
	h_crypt.init.mode      = CRYPT_MODE_ECB;//CRYPT_MODE_CBC
	h_crypt.init.key       = CRYPT_DES_KEYS_2;
	h_crypt.init.type      = DATA_32_BIT;
	h_crypt.crypt_cplt_cbk = crypt_complete;
	h_crypt.err_cplt_cbk   = error_complete;
	ald_crypt_init(&h_crypt);

	/* Start AES encrypt,result_buffer equal to expect_buffer */
	ald_crypt_write_key(&h_crypt,key_buf,KEY_4_LEN);
 	ald_crypt_encrypt(&h_crypt,(uint8_t*)plain_buffer, (uint8_t*)result_buffer, sizeof(plain_buffer));

	/* Start AES decrypt,result_buffer equal to expect_buffer */	
	ald_crypt_write_key(&h_crypt,key_buf,KEY_4_LEN);
 	ald_crypt_decrypt(&h_crypt,(uint8_t*)expect_buffer, (uint8_t*)result_buffer, sizeof(expect_buffer));	
}

/* des ecb key = 3 keys
 * input 	32 43 f6 a8 88 5a 30 8d
 * cipher key	2b 7e 15 16 28 ae d2 a6 ab f7 15 88 09 cf 4f 3c 2b 7e 15 16 28 ae d2 a6
 * refer result	bd 67 f6 a6 83 df b1 52
 */
void test_des_ecb_3key()
{
	uint32_t plain_buffer[2] = {0x3243F6A8, 0x885A308D};
	uint32_t result_buffer[2] = {0};
	uint32_t expect_buffer[2] = {0xbd67f6a6, 0x83dfb152};
	uint32_t key_buf[6]   = {0x2b7e1516, 0x28aed2a6, 0xabf71588, 0x09cf3c2b, 0x2b7e1516, 0x28aed2a6};

	/* Clear lpuart_handle_t structure */
	memset(&h_crypt, 0x0, sizeof(crypt_handle_t));

	/* Initialize crypt */
	h_crypt.perh           = CRYPT;
	h_crypt.init.mode      = CRYPT_MODE_ECB;//CRYPT_MODE_CBC
	h_crypt.init.key       = CRYPT_DES_KEYS_3;
	h_crypt.init.type      = DATA_32_BIT;
	h_crypt.crypt_cplt_cbk = crypt_complete;
	h_crypt.err_cplt_cbk   = error_complete;
	ald_crypt_init(&h_crypt);

	/* Start AES encrypt,result_buffer equal to expect_buffer */
	ald_crypt_write_key(&h_crypt,key_buf,KEY_6_LEN);
 	ald_crypt_encrypt(&h_crypt,(uint8_t*)plain_buffer, (uint8_t*)result_buffer, sizeof(plain_buffer));

	/* Start AES decrypt,result_buffer equal to expect_buffer */	
	ald_crypt_write_key(&h_crypt,key_buf,KEY_6_LEN);
 	ald_crypt_decrypt(&h_crypt,(uint8_t*)expect_buffer, (uint8_t*)result_buffer, sizeof(expect_buffer));	
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	/* Enable CRC mode clock */
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	/* des ecb mode 1 key case */
	test_des_ecb_1key();
	/* des ecb mode 2 key case */
	test_des_ecb_2key();
	/* des ecb mode 3 key case */
	test_des_ecb_3key();

	while (1) {
		ald_delay_ms(1000);
	}
}
/**
  * @}
  */
/**
  * @}
  */
