/*
 * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author        Notes
 * 2020-01-23     xxx        the first version
 */

#include <rthw.h>
#include <rtthread.h>
#include "board.h"
#include <ald_cmu.h>
#include <ald_gpio.h>
#ifdef LVGL_GUI
#include "lv_hal_tick.h"
#endif
/**
 * @addtogroup es32f3
 */

/*@{*/

/*******************************************************************************
* Function Name  : NVIC_Configuration
* Description    : Configures Vector Table base location.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_Configuration(void)
{
}

/*******************************************************************************
 * Function Name  : SystemClock_Configuration
 * Description    : Configures the System Clock.
 * Input          : None
 * Output         : None
 * Return         : None
 *******************************************************************************/
void  SystemClock_Config(void)
{	
	/* Initialize ALD */
	ald_cmu_init();
	
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
}

/*******************************************************************************
 * Function Name  : SysTick_Configuration
 * Description    : Configures the SysTick for OS tick.
 * Input          : None
 * Output         : None
 * Return         : None
 *******************************************************************************/
void  SysTick_Configuration(void)
{
    /* ticks = sysclk / RT_TICK_PER_SECOND */	
	SysTick_Config(ald_cmu_get_sys_clock() / SYSTICK_INTERVAL_1MS);
}

/**
 * This is the timer interrupt service routine.
 *
 */
void ald_systick_irq_cbk(void)
{
    /* enter interrupt */
    rt_interrupt_enter();

    rt_tick_increase();

#if defined LVGL_GUI
    lv_tick_inc(1);
#endif
	
    /* leave interrupt */
    rt_interrupt_leave();
}

/*@}*/
/**
 * This function will initial ES32F3 board.
 */
void rt_hw_board_init(void)
{
    /* NVIC Configuration */
    NVIC_Configuration();

    /*System Clock Configuration */
    SystemClock_Config();

    /* Configure the SysTick */
    SysTick_Configuration();

#ifdef RT_USING_HEAP
    rt_system_heap_init((void *)HEAP_BEGIN, (void *)HEAP_END);
#endif
#ifdef RT_USING_COMPONENTS_INIT
    rt_components_board_init();
#endif
#ifdef RT_USING_CONSOLE
    rt_console_set_device(RT_CONSOLE_DEVICE_NAME);
#endif
}

/**
 * This function will delay for some us.
 *
 * @param us the delay time of us
 */
void rt_hw_us_delay(rt_uint32_t us)
{
    unsigned int start, now, delta, reload, us_tick;
    start = SysTick->VAL;
    reload = SysTick->LOAD;
    us_tick = ald_cmu_get_sys_clock() / 1000000UL;
    do
    {
        now = SysTick->VAL;
        delta = start > now ? start - now : reload + start - now;
    }
    while (delta <  us_tick * us);
}
