/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Example DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include "main.h"

/* Rec macro */
#define REC_DISK	0
#define REC_PATH	"/Record"
extern uint8_t recbuf[512];
static void vs1053_record(void);

/**
  * @brief  System tick callback function
  * @retval Status
  */
void ald_systick_irq_cbk(void)
{
	static uint8_t cnt = 0;
	cnt++;
	
	if(cnt == 10) {
		cnt = 0;
		bsp_key_scan();
	}
	
	return;
}

/**
  * @brief  This is main function.
  * @retval Status
  */
int main()
{
	uint8_t tmp = 0;
	
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	/* Initialize peripherals */
	bsp_led_init();
	bsp_key_init();
	bsp_uart_init();
	bsp_audio_init();
	tmp = vs1053_read_chip_id();
	vs1053_set_volume(220);
	printf("VS1053 chip id is:%d\r\n", tmp);
	bsp_lcd_init();
	bsp_touch_init();
	/* Display */
	lcd_set_front_color(BLUE);
	lcd_show_string(80, 100, 210, 24, 24, "Record Test!\r\n");
	lcd_set_front_color(RED);
	lcd_show_string(30, 140, 320, 24, 16, "->Connect ESLinkII\r\n");
	lcd_show_string(30, 180, 320, 24, 16, "->Press K3 to record \r\n");
	lcd_show_string(30, 220, 320, 24, 16, "->Press K3 to stop record\r\n");
	
	while(1) {
		vs1053_record();	
	}
}

/**
  * @brief  vs1053 record function
  * @retval None
  */
void vs1053_record(void)
{
	static uint8_t kflg = 0;
	FRESULT result;
	FIL frec;
	uint32_t sectorsize = 0;
	uint16_t idx = 0;
	UINT len;
	wave_header_t wvh = {0};
	char name_buf[30];
	uint8_t keycode = 0;
	uint16_t wd = 0;
	FATFS fs;
	DIR dir_inf;
	char path[64];
	
	/* Open record directory */
	if (f_mount(&fs, "0:", 1) != FR_OK) {
		printf("Mount file system fail");
	}
	else 
		printf("Mount file system success \r\n");
	sprintf(path, "%d:%s", REC_DISK, REC_PATH);
	result = f_opendir(&dir_inf, path); 	
	if (result != FR_OK) {
		printf("Open record directory fail(%d)\r\n", result);
		result = f_mkdir(path);	
		if (result == FR_OK) {
			printf("Create record directory success\r\n");
		} else 
			return;
	} else {
		printf("Open record directory success\r\n");
	}
	/* Prepare for recording */
	bps_record_init();
	while(vs1053_read_reg(SCI_HDAT1) >> 8);
	/* Wait key to record */
	while(1) {
		/* Wait for key */
		keycode = bsp_get_key();
		if (keycode == KEY_DOWN_K3) {
			kflg++;
		}
		/* Record file name and write file header */
		if ((kflg == 0x01) && (recf.f.rec == 0))  {	            
			recf.f.rec = 1;
			record_check_new_name(name_buf);	              /* Get a new rec file name */
			lcd_show_string(30, 290, 240, 24, 16, name_buf + 10); /* Display rec file name */
			record_header_default_init(&wvh);                     /* Wave header default value */
			result = f_open(&frec,(const TCHAR*)name_buf, FA_CREATE_ALWAYS | FA_WRITE); /* Create wav file */
			if (result != FR_OK) {
				recf.flg = 0;
				kflg = 0;
				printf(" Rec wave file create fail\r\n");
			} else {
				result = f_write(&frec, (const void*)&wvh, sizeof(wave_header_t), &len);
				if (result == FR_OK)
					printf("Rec wave file create success and wave header write success\r\n");
			}
		}
		/* End record and add file size to the record header */	
		if (kflg == 0x02) {    
			kflg = 0;
			wvh.riff.chunk_size = sectorsize * 512 + 36;
			wvh.data.chunk_size = sectorsize * 512;
			f_lseek(&frec, 0);
			result = f_write(&frec, (const void*)&wvh, sizeof(wave_header_t), &len);
			if (result == FR_OK)
				printf("Rec wave header write ok\r\n");
				
			f_close(&frec);
			sectorsize = 0;
			recf.flg = 0;
			printf("Rec wave file finished\r\n");
			lcd_show_string(30, 260, 180, 16, 16, "Record end         ");
		}
		/* Record and write record data */
		if (recf.f.rec == 0x01) {
			wd = vs1053_read_reg(SCI_HDAT1);
			if ((wd >= 256) && (wd < 896)) {
				idx = 0;
				while(idx < 512) {
					wd = vs1053_read_reg(SCI_HDAT0);
					recbuf[idx++] = wd & 0xff;
					recbuf[idx++] = wd >> 8;
				}
				result = f_write(&frec,(const void*)recbuf, 512, &len);
				if (result == FR_OK) {
					printf("Write wave file ok, %d\r\n", ++sectorsize);
				} else {
					printf("Write wave file fail %d\r\n", result);
					recf.flg = 0;
					kflg = 0;
					break;
				}
				sectorsize++;
			}
			
			lcd_show_string(30, 260, 180, 16, 16, "Recording ...");
		}
	}
}





