/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Example DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include "main.h"

/* mp3 broadcast status struct */
typedef struct {
	uint8_t volume;			
	uint32_t progress;			
	uint8_t list_count;			
	uint8_t list_index;
	uint8_t next_flag : 1;
	uint8_t mute_on   : 1;
	uint8_t pause_en  : 1;
	uint8_t reserved  : 5;
} broadcast_status_t; 

/* mp3 file attribution struct */
typedef struct {
	uint32_t file_size;		
	char file_name[256];		
} mp3_file_t; 

/* mp3 macro */
#define MP3_PATH	"/Musics"	
#define MP3_DISK	0
#define SONG_LIST_MAX   10

/* broadcast global variable */
broadcast_status_t broadcast_status = {.volume = 200, 
                                       .progress = 0,
                                       .list_count = 0,
	                               .list_index = 0,
	                               .next_flag = 1,0};
mp3_file_t mp3_list[SONG_LIST_MAX];
FIL 	current_mp3_file;

/* Parivate function */
static void read_song_list(void);
static uint8_t mp3_process(void);
void mp3_player(void);

/**
  * @brief  System tick callback function
  * @retval Status
  */
void ald_systick_irq_cbk(void)
{
	static uint8_t cnt = 0;
	cnt++;
	if(cnt == 10) {
		cnt = 0;
		bsp_key_scan();
	}
	return;
}

/**
  * @brief  This is main function.
  * @retval Status
  */
int main()
{
	uint8_t temp = 0;
	
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	/* Initialize peripheral*/
	bsp_led_init();
	bsp_key_init();
	bsp_uart_init();
	bsp_audio_init();
	bsp_lcd_init();
	bsp_touch_init();
	/* Red vs1053 chip id */
	temp = vs1053_read_chip_id();
	printf("VS1053 chip id is:%d\r\n", temp);
	lcd_set_front_color(BLUE);
	lcd_show_string(80, 100, 210, 24, 24, "Audio Test!\r\n");
	lcd_set_front_color(RED);
	lcd_show_string(30, 140, 320, 24, 16, "->Connect ESLinkII\r\n");
	lcd_show_string(30, 180, 320, 24, 16, "->Put Musics directory in SDCard\r\n");
	lcd_show_string(30, 220, 320, 24, 16, "->Press K3 to switch next\r\n");
	while(1) {
		mp3_player();
	}
}

/**
  * @brief  Play SDcard mp3 file function
  * @retval None
  */
void mp3_player(void)
{
	uint8_t keycode;
	FRESULT result;
	TCHAR FileName[128];
	FATFS   sdfs;
	
	if (f_mount(&sdfs, "0:", 1) != FR_OK) {
		printf("Mount file system fail \r\n");
	}
	else 
		printf("Mount file system success \r\n");
	read_song_list();
	vs1053_spk_set(1);
	broadcast_status.volume = 220;
	vs1053_set_volume(broadcast_status.volume);
	vs1053_set_bass(0, 0, 15, 150);
	while(1) {
		if (broadcast_status.next_flag == 1) {
			broadcast_status.next_flag = 0;
			if (++broadcast_status.list_index >= broadcast_status.list_count) 
				broadcast_status.list_index = 0;
			f_close(&current_mp3_file);
			sprintf(FileName, "%d:%s/%s", MP3_DISK, MP3_PATH, mp3_list[broadcast_status.list_index].file_name);
			result = f_open(&current_mp3_file, FileName, FA_OPEN_EXISTING | FA_READ);
			if (result !=  FR_OK) {
				printf("Open mp3 file fail. %s\r\n", mp3_list[broadcast_status.list_index].file_name);
			}
			else {
				printf("Broadcasting: %s\r\n", mp3_list[broadcast_status.list_index].file_name);
			}
			vs1053_soft_reset();
		}
		/* read mp3 file and cast */
		if (mp3_process() == 1) 
			broadcast_status.next_flag = 1;
		/* key handler */
		keycode = bsp_get_key();
		if (keycode != KEY_NONE) {
			switch (keycode) {
			case KEY_DOWN_K3:
				broadcast_status.next_flag = 1;
				break;
			default: 
				break;
			}
		}
	}	
}

/**
  * @brief  Mp3 data process function
  * @retval Status
  */
static uint8_t mp3_process(void)
{
	uint32_t bw, i;
	char buf[32];

	if (vs1053_req_new_data() == 0) {
		f_read(&current_mp3_file, &buf, 32, &bw);
		if (bw <= 0) {
			return 1;
		}
		broadcast_status.progress += bw;
		vs1053_prewrite_data();
		for (i = 0; i < bw; i++) {
			vs1053_write_data(buf[i]);
		}
	}
	
	return 0;
}

/**
  * @brief  Get sdcard mp3 file list
  * @retval None
  */
static void read_song_list(void)
{
	FRESULT result;
	FILINFO mp3_inf;
	DIR mp3_dir;
	uint8_t namelen;
	char path[128];
	
	sprintf(path, "%d:%s", MP3_DISK, MP3_PATH);
	result = f_opendir(&mp3_dir, path); 	
	if (result != FR_OK) {
		printf("Open musics directory fail(%d)\r\n", result);
	} else printf("Open musics directory success\r\n");	
	broadcast_status.list_index = 0;
	broadcast_status.list_count = 0;	
	for(;;) {
		result = f_readdir(&mp3_dir, &mp3_inf); 		
		if ((result != FR_OK) || (mp3_inf.fname[0] == 0)) {
			break;
		}
		if (mp3_inf.fname[0] == '.') {	
			continue;
		}
		if (mp3_inf.fattrib != AM_DIR) {
			namelen = strlen(mp3_inf.fname);
			if (namelen >= 5) {
				if (memcmp(&mp3_inf.fname[namelen - 3], "mp3", 3) == 0) {
					strcpy(mp3_list[broadcast_status.list_count].file_name, mp3_inf.fname);
					mp3_list[broadcast_status.list_count].file_size = mp3_inf.fsize;
					printf("->[%d]:%-30s--%dbytes\r\n", broadcast_status.list_count + 1, \
                                               				    mp3_list[broadcast_status.list_count].file_name, \
									    mp3_list[broadcast_status.list_count].file_size);
					broadcast_status.list_count++;		
					if (broadcast_status.list_count > SONG_LIST_MAX) {
						printf("Mp3 files exceed the play list maximum number\r\n");
						break;
					}
				}
			}
		}
	}
	if (broadcast_status.list_count == 0) {
		printf("No mp3 file in the Musics directory \r\n");
	}
}



