/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Example DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"

/**
  * @brief  System tick callback function
  * @retval None
  */
void ald_systick_irq_cbk(void)
{
	static uint8_t cnt = 0;
	cnt++;
	
	if(cnt == 10) {
		cnt = 0;
		bsp_key_scan();
	}
	
	return;
}

/**
  * @brief  Fills buffer with user predefined data.
  * @param  pbuf: pointer on the buffer to fill
  * @param  buflen: size of the buffer to fill
  * @param  offset: first value to fill on the buffer
  * @retval None
  */
static void fill_buffer(uint16_t *pbuf, uint32_t buflen, uint16_t offset)
{
	uint16_t idx = 0;

	/* Put in global buffer different values */
	for (idx = 0; idx < buflen; idx++) {
		pbuf[idx] = idx + offset;
	}
}

/* Test Read/Write Buffers */
uint16_t txbuf[BUFFER_SIZE];
uint16_t rxbuf[BUFFER_SIZE];

/**
  * @brief  This is main function.
  * @retval Status
  */
int main()
{
	uint16_t tmp = 0;
	uint32_t *paddr = NULL;
	uint8_t keycode;
	
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	/* Initialize peripherals */
	bsp_led_init();
	bsp_key_init();
	bsp_uart_init();
	bsp_sram_pin_init();
	bsp_sram_init();
	bsp_lcd_init();
	bsp_touch_init();
	/* Display */
	lcd_set_front_color(BLUE);
	lcd_show_string(80, 100, 210, 24, 24, "SRAM Test!\r\n");
	lcd_set_front_color(RED);
	lcd_show_string(30, 140, 400, 24, 16, "->Connect ESLinkII\r\n");
	lcd_show_string(30, 180, 400, 24, 16, "->Press K2 to write data\r\n");
	lcd_show_string(30, 220, 440, 24, 16, "->Press K3 to read data\r\n");
	fill_buffer(txbuf, BUFFER_SIZE, tmp);
	paddr = (uint32_t *)SRAM_BANK_ADDR;
	
	while(1) {
		keycode = bsp_get_key();
		if (keycode != KEY_NONE) {
			switch (keycode) {
			case KEY_DOWN_K2:
				++tmp;
				fill_buffer(txbuf, BUFFER_SIZE, 4*tmp);
				if (ald_sram_write_16b(&hsram, paddr, txbuf, BUFFER_SIZE) == OK) {
					printf("Sram write success...\r\n");
				}
				break;
			case KEY_DOWN_K3:
				if (ald_sram_read_16b(&hsram, paddr, rxbuf, BUFFER_SIZE) == OK) {
					printf("Sram read success...\r\n");
					printf("Read value:0x%x 0x%x 0x%x 0x%x\r\n", rxbuf[0], rxbuf[1], rxbuf[2], rxbuf[3]);
				}
				break;
			default: 
				break;
			}
		}
	}
}



