/**
  *********************************************************************************
  *
  * @file    msgq.c
  * @brief   message queue management
  *
  * @version V1.0
  * @date    12 Mar 2020
  * @author  AE Team
  * @note   
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */ 

#include "msgq.h"
#include <string.h>
#include "utils.h"

static msg_t msg_pool[MSG_POOL_SIZE];
static msg_pool_t msg_env;


void msg_pool_init(void)
{
	int32_t i;

	msg_env.nr = 0;
	INIT_LIST_HEAD(&msg_env._free);

	for (i =0; i < MSG_POOL_SIZE; ++i) {
		__disable_irq();
		list_add_tail(&msg_pool[i].link, &msg_env._free);
		++msg_env.nr;
		__enable_irq();
	}

	return;
}

msg_t *msg_alloc(void)
{
	msg_t *msg;
	
	if (msg_env.nr == 0)
		return NULL;

	__disable_irq();
	msg = list_entry(msg_env._free.next, msg_t, link);
	list_del(msg_env._free.next);
	--msg_env.nr;
	__enable_irq();

	return msg;
}

void msg_free(msg_t *msg)
{
	__disable_irq();
	list_add_tail(&msg->link, &msg_env._free);
	++msg_env.nr;
	__enable_irq();

	return;
}

uint32_t msg_size_get(void)
{
	return msg_env.nr;
}

void msgq_init(msgq_t * msgq)
{
	if (msgq == NULL)
		return;

	msgq->nr = 0;
	INIT_LIST_HEAD(&msgq->list);

	return;
}

void msgq_enq(msgq_t *msgq, msg_t *msg)
{
	__disable_irq();
	list_add_tail(&msg->link, &msgq->list);
	++msgq->nr;
	__enable_irq();

	return;
}

msg_t *msgq_deq(msgq_t *msgq)
{
	msg_t *msg;

	if (msgq->nr == 0)
		return NULL;


	__disable_irq();
	msg = list_entry(msgq->list.next, msg_t, link);
	list_del(msgq->list.next);
	-- msgq->nr;
	__enable_irq();

	return msg;
}
