/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Nov 2020
  * @author  AE Team
  * @note   
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */ 

#include <string.h>
#include "main.h"
#include "usbd_printer.h"
#include "usbd_bulk.h"
#include "usbd_comp.h"

/** @addtogroup Projects_Examples_USB
  * @{
  */

/** @addtogroup Examples
  * @{
  */
env_t env;
uint8_t desc_buf[0x100];

uint32_t printer_rx_handle(void *data, uint32_t event, uint32_t value, void *p_data);
uint32_t printer_tx_handle(void *data, uint32_t event, uint32_t value, void *p_data);
uint32_t bulk_rx_handle(void *data, uint32_t event, uint32_t value, void *p_data);
uint32_t bulk_tx_handle(void *data, uint32_t event, uint32_t value, void *p_data);
/**
  * @brief Printer ID
  */
const uint8_t printer_id[] = "xxMANUFACTURER:Eastsoft; MODEL:NX-520; COMMEND SET:ESPON; CLASS:PRINTER";

/**
  * @brief String descriptor
  */
const uint8_t lang_desc[] = {
	4,
	USB_DTYPE_STRING,
	USBShort(USB_LANG_EN_US)
};

/**
  * @brief Manufact string
  */
const uint8_t manufact_str[] = {
	(17 + 1) * 2,   
	USB_DTYPE_STRING,
	'E', 0, 'a', 0, 's', 0, 't', 0, 's', 0, 'o', 0, 'f', 0, 't', 0,
	' ', 0, 'S', 0, 'h', 0, 'a', 0, 'n', 0, 'g', 0, 'h', 0, 'a', 0,
	'i', 0,
};

/**
  * @brief Product string
  */
const uint8_t product_str[] = {
	(16 + 1) * 2,
	USB_DTYPE_STRING,
	'E', 0, 'a', 0, 's', 0, 't', 0, 's', 0, 'o', 0, 'f', 0, 't', 0,
        ' ', 0, 'P', 0, 'r', 0, 'i', 0, 'n', 0, 't', 0, 'e', 0, 'r', 0,
};

/**
  * @brief Serial number string
  */
const uint8_t serial_num_str[] =
{
	(8 + 1) * 2,
	USB_DTYPE_STRING,
	'1', 0, '2', 0, '3', 0, '4', 0, '5', 0, '6', 0, '7', 0, '8', 0
};

/**
  * @brief Interface string
  */
const uint8_t data_interface_str[] =
{
	(14 + 1) * 2,
	USB_DTYPE_STRING,
	'B', 0, 'i', 0, '-', 0, 'd', 0, 'i', 0, 'r', 0, 'e', 0, 'c', 0,
	't', 0, 'i', 0,	'o', 0, 'n', 0, 'a', 0, 'l', 0, 
};

/**
  * @brief Configure string
  */
const uint8_t config_str[] =
{
	(23 + 1) * 2,
	USB_DTYPE_STRING,
	'B', 0, 'u', 0, 'l', 0, 'k', 0, ' ', 0, 'D', 0, 'a', 0, 't', 0,
	'a', 0, ' ', 0,	'C', 0, 'o', 0, 'n', 0, 'f', 0, 'i', 0, 'g', 0,
	'u', 0, 'r', 0, 'a', 0, 't', 0, 'i', 0, 'o', 0, 'n', 0
};

/**
  * @brief Product string
  */
const uint8_t product_str1[] = {
	(19 + 1) * 2,
	USB_DTYPE_STRING,
	'M', 0, 'a', 0, 's', 0, 's', 0, ' ', 0, 'S', 0, 't', 0, 'o', 0,
	'r', 0, 'a', 0, 'g', 0, 'e', 0, ' ', 0, 'D', 0, 'e', 0, 'v', 0,
	'i', 0, 'c', 0, 'e', 0
};

/**
  * @brief Interface string
  */
const uint8_t data_interface_str1[] =
{
	(19 + 1) * 2,
	USB_DTYPE_STRING,
	'B', 0, 'u', 0, 'l', 0, 'k', 0, ' ', 0, 'D', 0, 'a', 0, 't', 0,
	'a', 0, ' ', 0,	'I', 0, 'n', 0, 't', 0, 'e', 0, 'r', 0, 'f', 0,
	'a', 0, 'c', 0, 'e', 0
};

/**
  * @brief Configure string
  */
const uint8_t config_str1[] =
{
	(23 + 1) * 2,
	USB_DTYPE_STRING,
	'B', 0, 'u', 0, 'l', 0, 'k', 0, ' ', 0, 'D', 0, 'a', 0, 't', 0,
	'a', 0, ' ', 0,	'C', 0, 'o', 0, 'n', 0, 'f', 0, 'i', 0, 'g', 0,
	'u', 0, 'r', 0, 'a', 0, 't', 0, 'i', 0, 'o', 0, 'n', 0
};









/**
  * @brief String descriptor
  */
const uint8_t *const string_desc[] =
{
	lang_desc,
	manufact_str,
	product_str,
	serial_num_str,
	data_interface_str,
	config_str
};

const uint8_t *const string_desc1[] =
{
	lang_desc,
	manufact_str,
	product_str1,
	serial_num_str,
	data_interface_str1,
	config_str1
};

#define NUM_STRING_DESCRIPTORS (sizeof(string_desc) / sizeof(uint8_t *))

/**
  * @brief Printer device information
  */
usbd_printer_dev_t printer_device = {
	USB_VID_EASTSOFT_30CC,
	USB_PID_PRINTER,
	500,
	USB_CONF_ATTR_SELF_PWR,
	printer_rx_handle,
	&printer_device,
	printer_tx_handle,
	&printer_device,
	string_desc,
	NUM_STRING_DESCRIPTORS,
	printer_id,
	sizeof(printer_id),
};

/**
  * @brief BULK device information
  */
usbd_bulk_dev_t bulk_device = {
	USB_VID_EASTSOFT_30CC,
	USB_PID_BULK,
	500,
	USB_CONF_ATTR_SELF_PWR,
	bulk_rx_handle,
	&bulk_device,
	bulk_tx_handle,
	&bulk_device,
	string_desc1,
	NUM_STRING_DESCRIPTORS,
};

/**
  * @brief Composite entry array
  */
comp_entry_t entry[2];

/**
  * @brief Composite device information
  */
usbd_comp_dev_t comp_device =
{
	USB_VID_EASTSOFT_30CC,
	0x0012,
	500,
	USB_CONF_ATTR_SELF_PWR,
	0,
	string_desc,
	NUM_STRING_DESCRIPTORS,
	2,
	entry,
};


/**
  * @brief  Handle printer RX event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t printer_rx_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	uint8_t *p = (uint8_t *)p_data;

	switch (event) {
	case USB_EVENT_CONNECTED:
		env.flag = STATE_DEVICE_CONN;
		printf_e("\rPrinter is connected\r\n");
		break;
	case USB_EVENT_DISCONNECTED:
		printf_e("\rPrinter is disconnected\r\n");
		env.flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_RX_AVAILABLE:
		SET_BIT(env.flag, STATE_DEVICE_READ);
		break;
	case USBD_PRINTER_EVENT_GET_DEVICE_ID:     
		break;
	case USBD_PRINTER_EVENT_GET_PORT_STATUS:
		*p = env.printer_status;
		break;
	case USBD_PRINTER_EVENT_SOFT_RESET:
		/* Added software reset codes */
		break;
	case USB_EVENT_SUSPEND:
		/* Handle suspend */
		break;
	case USB_EVENT_ERROR:
		/* Handle error */
		break;
	default:
		break;
	}

	return 0;
}

/**
  * @brief  Handle printer TX event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t printer_tx_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_TX_COMPLETE:
		break;
	default:
		break;
	}

	return 0;
}


/**
  * @brief  Handle BULK RX event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t bulk_rx_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_CONNECTED:
		break;
	case USB_EVENT_DISCONNECTED:
		break;
	case USB_EVENT_SUSPEND:
		break;
	case USB_EVENT_RX_AVAILABLE:
//		env.flag = STATE_DEVICE_READ;
		break;
	case USB_EVENT_ERROR:
		break;
	default:
		break;
	}

	return 0;
}

/**
  * @brief  Handle BULK TX event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t bulk_tx_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_TX_COMPLETE:
		break;
	default:
		break;
	}

	return 0;
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	uint8_t nr;

	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

	/* Intialize clock and power */
	ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
	ald_rmu_reset_periperal(RMU_PERH_USB);
	/* Enable USB interrupt */
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
	ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, ENABLE);
	/* Disable High-Speed mode */
	ald_usb_high_speed_enable(false);

	/* Initialize ent */
	memset(&env, 0x0, sizeof(env_t));
	/* Initialize UART for debug */
	uart_stdio_init();
	/* Initialize pool */
	msg_pool_init();
	/* Initialize message queue */
	msgq_init(&env.mq);
	/* Initialize printer status */
	env.printer_status = USB_DEVICE_PRINTER_PORT_STATUS_DEFAULT_VALUE;

	/* Enable session */
	ald_usb_otg_session_request(true);
	/* Initialize USB */
	comp_device.device[0].inst = usbd_printer_init_comp(0, &printer_device, &entry[0]);
	comp_device.device[1].inst = usbd_bulk_init_comp(0, &bulk_device, &entry[1]);
	usbd_comp_init(0, &comp_device, 0x100, desc_buf);

	printf_e("System start...\r\n");

	while (1) {
		if (env.flag & STATE_DEVICE_READ) {	/* Handle data from host */
			printf_e("Receive a frame\r\n");
			CLEAR_BIT(env.flag, STATE_DEVICE_READ);
			nr = usbd_printer_rx_packet_avail(&printer_device);
			nr = usbd_printer_packet_read(&printer_device, env.rx_buf, nr, 1);
		}
	}
}

/**
  * @}
  */
/**
  * @}
  */
