/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"
#include "flow.h"


/** @addtogroup Projects_Examples_USB
  * @{
  */

/** @addtogroup Examples
  * @{
  */
env_t env;

uint32_t rx_handle(void *data, uint32_t event, uint32_t value, void *p_data);
uint32_t tx_handle(void *data, uint32_t event, uint32_t value, void *p_data);

/**
  * @brief Printer ID
  */
const uint8_t printer_id[] = "xxMANUFACTURER:Eastsoft; MODEL:NX-520; COMMEND SET:ESPON; CLASS:PRINTER";

/**
  * @brief String descriptor
  */
const uint8_t lang_desc[] = {
	4,
	USB_DTYPE_STRING,
	USBShort(USB_LANG_EN_US)
};

/**
  * @brief Manufact string
  */
const uint8_t manufact_str[] = {
	(17 + 1) * 2,   
	USB_DTYPE_STRING,
	'E', 0, 'a', 0, 's', 0, 't', 0, 's', 0, 'o', 0, 'f', 0, 't', 0,
	' ', 0, 'S', 0, 'h', 0, 'a', 0, 'n', 0, 'g', 0, 'h', 0, 'a', 0,
	'i', 0,
};

/**
  * @brief Product string
  */
const uint8_t product_str[] = {
	(16 + 1) * 2,
	USB_DTYPE_STRING,
	'E', 0, 'a', 0, 's', 0, 't', 0, 's', 0, 'o', 0, 'f', 0, 't', 0,
        ' ', 0, 'P', 0, 'r', 0, 'i', 0, 'n', 0, 't', 0, 'e', 0, 'r', 0,
};

/**
  * @brief Serial number string
  */
const uint8_t serial_num_str[] =
{
	(8 + 1) * 2,
	USB_DTYPE_STRING,
	'1', 0, '2', 0, '3', 0, '4', 0, '5', 0, '6', 0, '7', 0, '8', 0
};

/**
  * @brief Interface string
  */
const uint8_t data_interface_str[] =
{
	(14 + 1) * 2,
	USB_DTYPE_STRING,
	'B', 0, 'i', 0, '-', 0, 'd', 0, 'i', 0, 'r', 0, 'e', 0, 'c', 0,
	't', 0, 'i', 0,	'o', 0, 'n', 0, 'a', 0, 'l', 0, 
};

/**
  * @brief Configure string
  */
const uint8_t config_str[] =
{
	(23 + 1) * 2,
	USB_DTYPE_STRING,
	'B', 0, 'u', 0, 'l', 0, 'k', 0, ' ', 0, 'D', 0, 'a', 0, 't', 0,
	'a', 0, ' ', 0,	'C', 0, 'o', 0, 'n', 0, 'f', 0, 'i', 0, 'g', 0,
	'u', 0, 'r', 0, 'a', 0, 't', 0, 'i', 0, 'o', 0, 'n', 0
};

/**
  * @brief String descriptor
  */
const uint8_t *const string_desc[] =
{
	lang_desc,
	manufact_str,
	product_str,
	serial_num_str,
	data_interface_str,
	config_str
};

#define NUM_STRING_DESCRIPTORS (sizeof(string_desc) / sizeof(uint8_t *))


/**
  * @brief Printer device information
  */
usbd_printer_dev_t printer_device = {
	USB_VID_EASTSOFT_30CC,
	USB_PID_PRINTER,
	500,
	USB_CONF_ATTR_SELF_PWR,
	rx_handle,
	&printer_device,
	tx_handle,
	&printer_device,
	string_desc,
	NUM_STRING_DESCRIPTORS,
	printer_id,
	sizeof(printer_id),
};

/**
  * @brief  Handle printer RX event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t rx_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	uint8_t *p = (uint8_t *)p_data;

	switch (event) {
	case USB_EVENT_CONNECTED:
		env.flag = STATE_DEVICE_CONN;
		printf_e("\rPrinter is connected\r\n");
		break;
	case USB_EVENT_DISCONNECTED:
		printf_e("\rPrinter is disconnected\r\n");
		env.flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_RX_AVAILABLE:
#ifdef DEBUG_INFO
		printf_e("\rRecv a message:\r\n");
#endif
		flow_send_msg_to_rx_msgq();
		break;
	case USBD_PRINTER_EVENT_GET_DEVICE_ID:     
		break;
	case USBD_PRINTER_EVENT_GET_PORT_STATUS:
		*p = env.printer_status;
		break;
	case USBD_PRINTER_EVENT_SOFT_RESET:
		/* Added software reset codes */
		break;
	case USBD_PRINTER_EVENT_SYSTEM_RESET:
		/* Added system reset codes */
		ald_rmu_reset_periperal(RMU_PERH_CHIP);
		break;
	case USB_EVENT_SUSPEND:
		/* Handle suspend */
		break;
	case USB_EVENT_ERROR:
		/* Handle error */
		break;
	default:
		break;
	}

	return 0;
}

/**
  * @brief  Handle printer TX event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t tx_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_TX_COMPLETE:
		break;
	default:
		break;
	}

	return 0;
}

/**
  * @brief  Send data to PC.
  * @param  buf: Pointer to the buffer.
  * @param  len: Length of the data.
  * @retval None.
  */
void send_response_to_host(uint8_t *buf, uint32_t len)
{
	usbd_printer_packet_write(&printer_device, buf, len, 1);
	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Intialize clock and power */
	ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
	ald_rmu_reset_periperal(RMU_PERH_USB);
	/* Enable USB interrupt */
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
	ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, ENABLE);
	/* Disable High-Speed mode */
	ald_usb_high_speed_enable(false);

	/* Initialize ent */
	memset(&env, 0x0, sizeof(env_t));
	/* Initialize pool */
	msg_pool_init();
	/* Initialize message queue */
	msgq_init(&env.mq);
	/* Initialize printer status */
	env.printer_status = USB_DEVICE_PRINTER_PORT_STATUS_DEFAULT_VALUE;

	/* Enable session */
	ald_usb_otg_session_request(true);
	/* Initialize USB */
	usbd_printer_init(0, &printer_device);

	printf_e("System start...\r\n");

	/* Initialize flow */
	flow_init();
	return 0;
}

/**
  * @}
  */
/**
  * @}
  */
