/**
  *********************************************************************************
  *
  * @file    flow.c
  * @brief   Data flow management
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include <rtthread.h>
#include "flow.h"
#include "msgq.h"
#include "msg_type.h"
#include "msg_parse.h"
#include "ald_iap.h"
#include "ald_gpio.h"
#include "uart_stdio.h"
#include "usbd_printer.h"

#define THREAD_STACK_SIZE	1024
#define THREAD_PRIORITY_RX	6	/* PC-->ME */
#define THREAD_PRIORITY_TX	8	/* PC<--ME */

static struct rt_semaphore *rx_sem;
static struct rt_semaphore *tx_sem;
static msgq_t rx_msgq;
static msgq_t tx_msgq;


/** @brief Handles the message from the host.
  * @param arg: Parameter of the task.
  * @retval None.
  */
static void thread_entry_rx(void* arg)
{
	rt_err_t ret;
	msg_t *msg;

	while (1) {
		ret = rt_sem_take(rx_sem, RT_WAITING_FOREVER);
		if (ret != RT_EOK) {
			rt_kprintf("ERROR: RX sem error!\n");
			return;
		}

		if ((msg = msgq_deq(&rx_msgq)) != RT_NULL) {
			msg_parse(msg->buf, msg->len);
			msg_free(msg);
		}
		else {
			rt_kprintf("ERR0R: RX msgq is empty!\n");
		}
	}
}

/** @brief Handles the message from the device[machine].
  * @param arg: Parameter of the task.
  * @retval None.
  */
static void thread_entry_tx(void* arg)
{
	rt_err_t ret;
	msg_t *msg;

	while (1) {
		ret = rt_sem_take(tx_sem, RT_WAITING_FOREVER);
		
		if (ret != RT_EOK) {
			rt_kprintf("ERROR: TX sem error!\n");
			return;
		}

		if ((msg = msgq_deq(&tx_msgq)) != RT_NULL) {
#ifdef DEBUG_INFO
			rt_kprintf("Send a message[%d]\n", msg->len);
#endif
			send_response_to_host(msg->buf, msg->len);
			msg_free(msg);
		}
		else {
			rt_kprintf("ERR0R: TX msgq is empty!\n");
		}
	}
}

/** @brief Receive message from the host.
  * @retval Status.
  */
int flow_send_msg_to_rx_msgq(void)
{
	msg_t *msg;

	if ((msg = msg_alloc()) == NULL) {
		rt_kprintf("ERR0R: Message pool is empty!\n");
		return -1;
	}

	msg->len = usbd_printer_rx_packet_avail(&printer_device);
	msg->len = usbd_printer_packet_read(&printer_device, msg->buf, msg->len, 1);

	msgq_enq(&rx_msgq, msg);
	rt_sem_release(rx_sem);

	return 0;
}

/** @brief Send message to the host.
  * @param buf: Pointer to the data.
  * @param size: Size of the data.
  * @retval Status.
  */
int flow_send_msg_to_tx_msgq(uint8_t *buf, uint32_t size)
{
	msg_t *msg;

	if ((msg = msg_alloc()) == NULL) {
		rt_kprintf("ERR0R: Message pool is empty!\n");
		return -1;
	}

	msg->len = size;
	memcpy(msg->buf, buf, size);
	msgq_enq(&tx_msgq, msg);
	rt_sem_release(tx_sem);

	return 0;
}

/** @brief Check the XOR of the message.
  * @param buf: Pointer to the data.
  * @param size: Size of the data.
  * @param chk: Value of the XOR.
  * @retval Status.
  */
ald_status_t flow_msg_xor_check(uint8_t *buf, uint8_t len, uint8_t chk)
{
	uint8_t value = buf[0];
	uint8_t i = 0;

	if ((buf == NULL) || (len == 0))
		return ERROR;

	for (i = 1; i < len; i++)
		value = buf[i] ^ value;

	if (value == chk)
		return OK;

	return ERROR;
}

/** @brief Calculates the XOR of the message.
  * @param buf: Pointer to the data.
  * @param size: Size of the data.
  * @param chk: Value of the XOR.
  * @retval Status.
  */
ald_status_t flow_msg_xor_calc(uint8_t *buf, uint8_t len, uint8_t *chk)
{
	uint8_t value = buf[0];
	uint8_t i = 0;
	
	if ((buf == NULL) || (len == 0))
		return ERROR;

	for (i = 1; i < len; i++)
		value = (buf[i] ^ value);	

	*chk = value;

	return OK;
}

/** @brief Initialize the flow.
  * @retval Status.
  */
int flow_init(void)
{
	rt_thread_t tid1, tid2;

	rx_sem = rt_sem_create("rx_sem", 0, RT_IPC_FLAG_FIFO);
	if (rx_sem == RT_NULL)
		return -1;

	tx_sem = rt_sem_create("tx_sem", 0, RT_IPC_FLAG_FIFO);
	if (tx_sem == RT_NULL)
		return -1;

	tid1 = rt_thread_create("t_recv", thread_entry_rx, (void*)0,
			THREAD_STACK_SIZE, THREAD_PRIORITY_RX, 2);
	if (tid1 != RT_NULL)
		rt_thread_startup(tid1);

	tid2 = rt_thread_create("t_send", thread_entry_tx, (void*)0,
			THREAD_STACK_SIZE, THREAD_PRIORITY_TX, 2);
	if (tid2 != RT_NULL)
		rt_thread_startup(tid2);

	if (tid1 == RT_NULL || tid2 == RT_NULL)
		return -2;

	msgq_init(&rx_msgq);
	msgq_init(&tx_msgq);

	return 0;
}
