/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Board Initialization
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <rthw.h>
#include <rtthread.h>
#include <string.h>
#include "ald_cmu.h"
#include "ald_gpio.h"
#include "ald_uart.h"
#include "uart_stdio.h"
#include "board.h"


struct rt_device __uart1;
rt_err_t uart4_init(rt_device_t dev);
rt_err_t uart4_open(rt_device_t dev, rt_uint16_t oflag);
rt_size_t uart4_read(rt_device_t dev, rt_off_t pos, void *buffer, rt_size_t size);
rt_size_t uart4_write(rt_device_t dev, rt_off_t pos, const void *buffer, rt_size_t size);

/**
 * This function will initial STM32 board.
 */
void rt_hw_board_init(void)
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock at 72MHz */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

	/* Enable all peripherals */
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_UART0, ENABLE);
	/* Set SysTick as 10ms */
	ald_systick_interval_select(SYSTICK_INTERVAL_10MS);
	
	memset(&__uart1, 0x0, sizeof(struct rt_device));

	__uart1.init  = uart4_init;
	__uart1.open  = uart4_open;
	__uart1.read  = uart4_read;
	__uart1.write = uart4_write;
	rt_device_register(&__uart1, "uart1", RT_DEVICE_FLAG_RDWR);

#ifdef RT_USING_HEAP
	rt_system_heap_init((void *)HEAP_BEGIN, (void *)HEAP_END);
#endif
#ifdef RT_USING_COMPONENTS_INIT
	rt_components_board_init();
#endif
#ifdef RT_USING_CONSOLE
	rt_console_set_device(RT_CONSOLE_DEVICE_NAME);
#endif
}

/**
 * This is the systick interrupt service routine.
 *
 */
void rt_tick_handler(void)
{
	/* enter interrupt */
	rt_interrupt_enter();
	rt_tick_increase();
	/* leave interrupt */
	rt_interrupt_leave();
}

/**
  * @brief  Initialize the UART1
  * @retval None
  */
rt_err_t uart4_init(rt_device_t dev)
{
	uart_stdio_init();
	return 0;
}

rt_err_t uart4_open(rt_device_t dev, rt_uint16_t oflag)
{
	return uart4_init(dev);
}

rt_size_t uart4_read(rt_device_t dev, rt_off_t pos, void *buffer, rt_size_t size)
{
	ald_uart_recv_n_lock(&h_uart, (uint8_t *)buffer, size, 10);
	return size;
}


rt_size_t uart4_write(rt_device_t dev, rt_off_t pos, const void *buffer, rt_size_t size)
{
	ald_uart_send_n_lock(&h_uart, (uint8_t *)buffer, size, 1000);
	return size;
}
