/**
  *********************************************************************************
  *
  * @file    msgq.c
  * @brief   message queue management
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include "msgq.h"
#include <string.h>
#include "utils.h"

static msg_t msg_pool[MSG_POOL_SIZE];
static msg_pool_t msg_env;


void msg_pool_init(void)
{
	int32_t i;

	msg_env.nr = 0;
	INIT_LIST_HEAD(&msg_env._free);

	for (i =0; i < MSG_POOL_SIZE; ++i) {
		__disable_irq();
		list_add_tail(&msg_pool[i].link, &msg_env._free);
		++msg_env.nr;
		__enable_irq();
	}

	return;
}

msg_t *msg_alloc(void)
{
	msg_t *msg;
	
	if (msg_env.nr == 0)
		return NULL;

	__disable_irq();
	msg = list_entry(msg_env._free.next, msg_t, link);
	list_del(msg_env._free.next);
	--msg_env.nr;
	__enable_irq();

	return msg;
}

void msg_free(msg_t *msg)
{
	__disable_irq();
	list_add_tail(&msg->link, &msg_env._free);
	++msg_env.nr;
	__enable_irq();

	return;
}

uint32_t msg_size_get(void)
{
	return msg_env.nr;
}

void msgq_init(msgq_t * msgq)
{
	if (msgq == NULL)
		return;

	msgq->nr = 0;
	INIT_LIST_HEAD(&msgq->list);

	return;
}

void msgq_enq(msgq_t *msgq, msg_t *msg)
{
	__disable_irq();
	list_add_tail(&msg->link, &msgq->list);
	++msgq->nr;
	__enable_irq();

	return;
}

msg_t *msgq_deq(msgq_t *msgq)
{
	msg_t *msg;

	if (msgq->nr == 0)
		return NULL;


	__disable_irq();
	msg = list_entry(msgq->list.next, msg_t, link);
	list_del(msgq->list.next);
	-- msgq->nr;
	__enable_irq();

	return msg;
}
