/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_USB
  * @{
  */

/** @addtogroup Examples
  * @{
  */
env_t env;

void uart0_init(void);
uint32_t contrl_handle(void *data, uint32_t event, uint32_t value, void *p_data);
uint32_t rx_handle(void *data, uint32_t event, uint32_t value, void *p_data);
uint32_t tx_handle(void *data, uint32_t event, uint32_t value, void *p_data);

/**
  * @brief String descriptor
  */
const uint8_t lang_desc[] =
{
	4,
	USB_DTYPE_STRING,
	USBShort(USB_LANG_EN_US)
};

/**
  * @brief Manufact string
  */
const uint8_t manufact_str[] = {
	(17 + 1) * 2,   
	USB_DTYPE_STRING,
	'E', 0, 'a', 0, 's', 0, 't', 0, 's', 0, 'o', 0, 'f', 0, 't', 0,
	' ', 0, 'S', 0, 'h', 0, 'a', 0, 'n', 0, 'g', 0, 'h', 0, 'a', 0,
	'i', 0,
};

/**
  * @brief Product string
  */
const uint8_t product_str[] = {
	(16 + 1) * 2,
	USB_DTYPE_STRING,
	'V', 0, 'i', 0, 'r', 0, 't', 0, 'u', 0, 'a', 0, 'l', 0, ' ', 0,
	'C', 0, 'O', 0, 'M', 0, ' ', 0, 'P', 0, 'o', 0, 'r', 0, 't', 0
};

/**
  * @brief Serial number string
  */
const uint8_t serial_num_str[] =
{
	(8 + 1) * 2,
	USB_DTYPE_STRING,
	'1', 0, '2', 0, '3', 0, '4', 0, '5', 0, '6', 0, '7', 0, '8', 0
};

/**
  * @brief Interface string
  */
const uint8_t data_interface_str[] =
{
	(21 + 1) * 2,
	USB_DTYPE_STRING,
	'A', 0, 'C', 0, 'M', 0, ' ', 0, 'C', 0, 'o', 0, 'n', 0, 't', 0,
	'r', 0, 'o', 0, 'l', 0, ' ', 0, 'I', 0, 'n', 0, 't', 0, 'e', 0,
	'r', 0, 'f', 0, 'a', 0, 'c', 0, 'e', 0
};

/**
  * @brief Configure string
  */
const uint8_t config_str[] =
{
	(26 + 1) * 2,
	USB_DTYPE_STRING,
	'S', 0, 'e', 0, 'l', 0, 'f', 0, ' ', 0, 'P', 0, 'o', 0, 'w', 0,
	'e', 0, 'r', 0, 'e', 0, 'd', 0, ' ', 0, 'C', 0, 'o', 0, 'n', 0,
	'f', 0, 'i', 0, 'g', 0, 'u', 0, 'r', 0, 'a', 0, 't', 0, 'i', 0,
	'o', 0, 'n', 0
};

/**
  * @brief String descriptor
  */
const uint8_t *const string_desc[] =
{
	lang_desc,
	manufact_str,
	product_str,
	serial_num_str,
	data_interface_str,
	config_str
};

#define NUM_STRING_DESCRIPTORS (sizeof(string_desc) / sizeof(uint8_t *))


/**
  * @brief CDC device information
  */
usbd_cdc_dev_t cdc_device = {
	USB_VID_EASTSOFT_30CC,
	USB_PID_SERIAL,
	500,
	USB_CONF_ATTR_SELF_PWR,
	contrl_handle,
	(void *)&cdc_device,
	rx_handle,
	(void *)&cdc_device,
	tx_handle,
	(void *)&cdc_device,
	string_desc,
	NUM_STRING_DESCRIPTORS
};

/**
  * @brief  Callback function
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None
  */
void uart0_tx_cplt(struct uart_handle_s *arg)
{
	env.flag_uart_tx = STATE_UART_TX_CPLT;
	return;
}

/**
  * @brief  Callback function
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None
  */
void uart0_rx_cplt(struct uart_handle_s *arg)
{
	msg_t *msg;

	if ((msg = msg_alloc()) == NULL) {
		printf_e("Memory out!\r\n");
		return;
	}

	msg->len = arg->rx_count;
	memcpy(msg->buf, env.tx_buf, msg->len);
	msgq_enq(&env.mq, msg);
	ald_uart_recv_frame_by_it(&env.h_uart0, env.tx_buf, 64, 40);

	return;
}

/**
  * @brief  Callback function
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None
  */
void uart0_error(struct uart_handle_s *arg)
{
	uart0_init();
	return;
}

/**
  * @brief  Initialize the UART0
  * @retval None
  */
void uart0_init(void)
{
	gpio_init_t x;

	ald_cmu_perh_clock_config(CMU_PERH_UART0, ENABLE);
	ald_mcu_irq_config(UART0_IRQn, 1, 1, ENABLE);

	/* Initialize tx pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_10, &x);

	/* Initialize rx pin */
	x.mode  = GPIO_MODE_INPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_11, &x);

	/* Initialize uart */
	env.h_uart0.perh             = UART0;
	env.h_uart0.init.baud        = 115200;
	env.h_uart0.init.word_length = UART_WORD_LENGTH_8B;
	env.h_uart0.init.stop_bits   = UART_STOP_BITS_1;
	env.h_uart0.init.parity      = UART_PARITY_NONE;
	env.h_uart0.init.mode        = UART_MODE_UART;
	env.h_uart0.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	env.h_uart0.tx_cplt_cbk      = uart0_tx_cplt;
	env.h_uart0.rx_cplt_cbk      = uart0_rx_cplt;
	env.h_uart0.error_cbk        = uart0_error;
	ald_uart_init(&env.h_uart0);

	return;
}

/**
  * @brief  Get UART's parameters.
  * @param  lc: Pointer to line_coding_t structure.
  * @retval None
  */
void get_line_coding(line_coding_t *lc)
{
	printf_e("\rGet param\n\r");
	lc->rate     = env.s_param.rate;
	lc->data_bit = env.s_param.data_bit;
	lc->parity   = env.s_param.parity;
	lc->stop     = env.s_param.stop;
	return;
}

/**
  * @brief  Set UART's parameters.
  * @param  lc: Pointer to line_coding_t structure.
  * @retval None
  */
void set_line_coding(line_coding_t *lc)
{
	env.s_param.rate     = lc->rate;
	env.s_param.data_bit = lc->data_bit;
	env.s_param.parity   = lc->parity;
	env.s_param.stop     = lc->stop == 0 ? 1 : lc->stop;

	env.h_uart0.perh             = UART0;
	env.h_uart0.init.baud        = env.s_param.rate;
	env.h_uart0.init.word_length = UART_WORD_LENGTH_8B;
	env.h_uart0.init.stop_bits   = env.s_param.stop == 1 ? UART_STOP_BITS_1 : UART_STOP_BITS_2;
	env.h_uart0.init.mode        = UART_MODE_UART;
	env.h_uart0.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	env.h_uart0.tx_cplt_cbk      = uart0_tx_cplt;
	env.h_uart0.rx_cplt_cbk      = uart0_rx_cplt;
	env.h_uart0.error_cbk        = uart0_error;

	switch (env.s_param.parity) {
	case 0:
		env.h_uart0.init.parity = UART_PARITY_NONE;
		break;
	case 1:
		env.h_uart0.init.parity = UART_PARITY_ODD;
		break;
	case 2:
		env.h_uart0.init.parity = UART_PARITY_EVEN;
		break;
	default:
		env.h_uart0.init.parity = UART_PARITY_NONE;
		break;
	}

	ald_uart_init(&env.h_uart0);
	ald_uart_recv_frame_by_it(&env.h_uart0, env.tx_buf, 64, 40);

	printf_e("Rate: %d\r\n", env.s_param.rate);
	printf_e("DataBit: %d\r\n", env.s_param.data_bit);
	printf_e("Parity: %d\r\n", env.s_param.parity);
	printf_e("StopBit: %d\r\n", env.s_param.stop);
	return;
}

/**
  * @brief  Control UART's status.
  * @param  value: Value of UART's status.
  * @retval None
  */
void set_control_line_state(uint32_t value)
{
	printf_e("Value: %d\r\n", value);
	return;
}

/**
  * @brief  Send break.
  * @param  flag: Flags.
  * @retval None
  */
void send_break(uint8_t flag)
{
	printf_e("Send Break: %d\r\n", flag);
	return;
}

/**
  * @brief  Handle CDC control event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t contrl_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_CONNECTED:
		env.flag = STATE_DEVICE_CONN;
		break;
	case USB_EVENT_DISCONNECTED:
		break;
	case USB_EVENT_SUSPEND:
		printf_e("Suspend!\r\n");
		env.flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_RESUME:
		break;
	case USBD_CDC_EVENT_GET_LINE_CODING:
		get_line_coding(p_data);
		break;
	case USBD_CDC_EVENT_SET_LINE_CODING:
		set_line_coding(p_data);
		break;
	case USBD_CDC_EVENT_SET_CTRL_STATE:
		set_control_line_state(value);
		break;
	case USBD_CDC_EVENT_SEND_BREAK:
		send_break(1);
		break;
	case USBD_CDC_EVENT_CLEAR_BREAK:
		send_break(0);
		break;

	default:
		break;
	}

	return 0;
}

/**
  * @brief  Handle CDC RX event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t rx_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_RX_AVAILABLE:
		env.flag = STATE_DEVICE_READ;
		break;
	case USB_EVENT_DATA_REMAINING:
		break;
	case USB_EVENT_REQUEST_BUFFER:
		break;
	default:
		break;
	}

	return 0;
}

/**
  * @brief  Handle CDC TX event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t tx_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_TX_COMPLETE:
		env.flag_usb_tx = STATE_USB_TX_CPLT;
		break;
	default:
		break;
	}

	return 0;
}

/**
  * @brief  Send data by interrupt via UART0
  * @param  size: Size of the data
  * @retval None
  */
void uart_send_by_it(uint32_t size)
{
	env.h_uart0.tx_buf   = env.rx_buf;
	env.h_uart0.tx_size  = size;
	env.h_uart0.tx_count = 0;

	SET_BIT(UART0->ICR, UART_ICR_TFTH_MSK);
	WRITE_REG(UART0->IER,UART_IT_TFTH);
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	uint8_t nr;
	msg_t *msg;

	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_96M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 96000000);

	memset(&env, 0x0, sizeof(env_t));

	uart_stdio_init();
	uart0_init();
	msg_pool_init();
	msgq_init(&env.mq);

	env.s_param.rate     = 115200;
	env.s_param.data_bit = 8;
	env.s_param.parity   = USB_CDC_PARITY_NONE;
	env.s_param.stop     = USB_CDC_STOP_BITS_1;

	ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
	ald_rmu_reset_periperal(RMU_PERH_USB);
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
	ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, ENABLE);
	ald_usb_high_speed_enable(false);
	ald_usb_otg_session_request(true);
	usbd_cdc_init(0, &cdc_device);

	env.flag_usb_tx  = STATE_USB_TX_CPLT;
	env.flag_uart_tx = STATE_UART_TX_CPLT;
	printf_e("System start...\r\n");

	while (1) {
		if ((env.flag_uart_tx == STATE_UART_TX_CPLT) && (env.flag == STATE_DEVICE_READ))   {
			nr = usbd_cdc_rx_packet_avail(&cdc_device);
			nr = usbd_cdc_packet_read(&cdc_device, env.rx_buf, nr, 1);
			env.flag_uart_tx = STATE_BUSY;
			uart_send_by_it(nr);
			env.flag = STATE_DEVICE_READY;
		}

		if (env.mq.nr && env.flag_usb_tx == STATE_USB_TX_CPLT) {
			env.flag_usb_tx = STATE_BUSY;
			msg = msgq_deq(&env.mq);
			while (!(usbd_cdc_tx_packet_avail(&cdc_device)));
			usbd_cdc_packet_write(&cdc_device, msg->buf, msg->len, 1);
			msg_free(msg);
		}
	}
}

/**
  * @}
  */
/**
  * @}
  */
