/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"
#include "usbh_bulk.h"
#include "usbd_bulk.h"


/** @addtogroup Projects_Examples_USB
  * @{
  */

/** @addtogroup Examples
  * @{
  */

#define STATE_DEVICE_NO		0x01
#define STATE_DEVICE_ENUM	0x02
#define STATE_DEVICE_READY	0x04
#define STATE_DEVICE_UNKNOWN	0x08
#define STATE_DEVICE_READ	0x10
#define STATE_POWER_FAULT	0x20

#define USBH_POOL_LEN	128

uint32_t rx_nr;
env_t env;
uint8_t tx_buf[512];
uint8_t rx_buf[512];
uint8_t usbh_pool[USBH_POOL_LEN];
uint32_t _bulk_host_flag = STATE_DEVICE_NO;
uint32_t _bulk_dev_flag  = STATE_DEVICE_NO;
usbh_bulk_inst_t *inst;

void usb_hcd_event(void *pdata);
DECLARE_EVENT_DRIVER(usb_event_driver, 0, 0, usb_hcd_event);
static usbh_class_driver_t const * const __host_driver[] = {
	&__usbh_bulk_driver,
	&usb_event_driver
};
static const uint32_t __nr_host_driver = sizeof(__host_driver) / sizeof(usbh_class_driver_t *);

uint32_t rx_handle(void *data, uint32_t event, uint32_t value, void *p_data);
uint32_t tx_handle(void *data, uint32_t event, uint32_t value, void *p_data);

/**
  * @brief  Initializate pin of USB.
  * @retval None
  */
void usb_pin_init(void)
{
	gpio_init_t x;

	/* Initialize vbus pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_1;
	ald_gpio_init(GPIOB, GPIO_PIN_15, &x);
	ald_gpio_write_pin(GPIOB, GPIO_PIN_15, 1);

	return;
}

/**
  * @brief String descriptor
  */
const uint8_t lang_desc[] =
{
	4,
	USB_DTYPE_STRING,
	USBShort(USB_LANG_EN_US)
};

/**
  * @brief Manufact string
  */
const uint8_t manufact_str[] = {
	(17 + 1) * 2,   
	USB_DTYPE_STRING,
	'E', 0, 'a', 0, 's', 0, 't', 0, 's', 0, 'o', 0, 'f', 0, 't', 0, ' ', 0,
	'S', 0, 'h', 0, 'a', 0, 'n', 0, 'g', 0, 'h', 0, 'a', 0, 'i', 0
};

/**
  * @brief Product string
  */
const uint8_t product_str[] = {
	(19 + 1) * 2,
	USB_DTYPE_STRING,
	'M', 0, 'a', 0, 's', 0, 's', 0, ' ', 0, 'S', 0, 't', 0, 'o', 0,
	'r', 0, 'a', 0, 'g', 0, 'e', 0, ' ', 0, 'D', 0, 'e', 0, 'v', 0,
	'i', 0, 'c', 0, 'e', 0
};

/**
  * @brief Serial number string
  */
const uint8_t serial_num_str[] =
{
	(8 + 1) * 2,
	USB_DTYPE_STRING,
	'1', 0, '2', 0, '3', 0, '4', 0, '5', 0, '6', 0, '7', 0, '8', 0
};

/**
  * @brief Interface string
  */
const uint8_t data_interface_str[] =
{
	(19 + 1) * 2,
	USB_DTYPE_STRING,
	'B', 0, 'u', 0, 'l', 0, 'k', 0, ' ', 0, 'D', 0, 'a', 0, 't', 0,
	'a', 0, ' ', 0,	'I', 0, 'n', 0, 't', 0, 'e', 0, 'r', 0, 'f', 0,
	'a', 0, 'c', 0, 'e', 0
};

/**
  * @brief Configure string
  */
const uint8_t config_str[] =
{
	(23 + 1) * 2,
	USB_DTYPE_STRING,
	'B', 0, 'u', 0, 'l', 0, 'k', 0, ' ', 0, 'D', 0, 'a', 0, 't', 0,
	'a', 0, ' ', 0,	'C', 0, 'o', 0, 'n', 0, 'f', 0, 'i', 0, 'g', 0,
	'u', 0, 'r', 0, 'a', 0, 't', 0, 'i', 0, 'o', 0, 'n', 0
};

/**
  * @brief String descriptor
  */
const uint8_t *const string_desc[] =
{
	lang_desc,
	manufact_str,
	product_str,
	serial_num_str,
	data_interface_str,
	config_str
};

#define NUM_STRING_DESCRIPTORS (sizeof(string_desc) / sizeof(uint8_t *))

/**
  * @brief BULK device information
  */
usbd_bulk_dev_t bulk_device = {
	USB_VID_EASTSOFT_30CC,
	USB_PID_BULK,
	500,
	USB_CONF_ATTR_SELF_PWR,
	rx_handle,
	&bulk_device,
	tx_handle,
	&bulk_device,
	string_desc,
	NUM_STRING_DESCRIPTORS,
};

/**
  * @brief  Handle BULK RX event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t rx_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_CONNECTED:
		_bulk_dev_flag = STATE_DEVICE_ENUM;
		break;
	case USB_EVENT_DISCONNECTED:
		_bulk_dev_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_SUSPEND:
		_bulk_dev_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_RX_AVAILABLE:
		_bulk_dev_flag = STATE_DEVICE_READ;
		break;
	case USB_EVENT_ERROR:
		_bulk_dev_flag = STATE_POWER_FAULT;
		break;
	default:
		break;
	}

	return 0;
}

/**
  * @brief  Handle BULK TX event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t tx_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_TX_COMPLETE:
		_bulk_dev_flag = STATE_DEVICE_READY;
		break;
	default:
		break;
	}

	return 0;
}


/**
  * @brief  BULK callback.
  * @param  inst: BULK instance.
  * @param  event: Type of the event.
  * @param  data: message of the event.
  * @retval None
  */
void usr_bulk_cbk(usbh_bulk_inst_t *inst, uint32_t event, void *data)
{
	switch (event) {
	case USBH_BULK_EVENT_OPEN:
		printf_e("\rHost device enum\n\r");
		_bulk_host_flag = STATE_DEVICE_ENUM;
		break;
	case USBH_BULK_EVENT_CLOSE:
		printf_e("\rHost device close\n\r");
		_bulk_host_flag = STATE_DEVICE_NO;
		break;
	default:
		break;
	}
}

/**
  * @brief  Handle HCD event.
  * @param  data: Parameter of the event.
  * @retval None
  */
void usb_hcd_event(void *data)
{
	event_info_t *info = (event_info_t *)data;

	switch (info->event) {
	case USB_EVENT_CONNECTED:
		/* Device is connect */
		printf_e("\rHost Device Connected\n\r");
		break;
	case USB_EVENT_DISCONNECTED:
		/* Device is disconnect */
		printf_e("\rHost Device Disconnected\n\r");
		env.flag = 0;
		_bulk_host_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_POWER_FAULT:
		printf_e("\rHOST POWER FAULT\n\r");
		_bulk_host_flag = STATE_POWER_FAULT;
		break;
	default:
		break;
	}
}

/**
  * @brief  Callback of the mode change.
  * @param  index: Index of the USB controller, always 0.
  * @param  mode: New mode.
  * @retval None
  */
void usb_otg_mode_cbk(uint32_t index, usb_mode_t mode)
{
	assert_param(index == 0);

	switch (mode) {
	case USB_LIB_MODE_HOST:
		printf_e("\rEnter host mode\n\r");	
		break;
	
	case USB_LIB_MODE_DEVICE:
		printf_e("\rEnter device mode\n\r");
		usbd_bulk_init(0, &bulk_device);
		/* Close vbus */
		ald_gpio_write_pin(GPIOB, GPIO_PIN_15, 0);
		break;
	
	case USB_LIB_MODE_NONE:
		printf_e("\rExit host/device mode\n\r");
		break;
	
	default:
		break;
	}

	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
  	uint32_t i, ret, len;

	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

	memset(&env, 0x0, sizeof(env_t));
	memset(tx_buf, 0x11, 64);
	memset(tx_buf + 64, 0x22, 64);
	memset(tx_buf + 64 * 2, 0x33, 64);
	memset(tx_buf + 64 * 3, 0x44, 64);
	memset(tx_buf + 64 * 4, 0x55, 64);
	memset(tx_buf + 64 * 5, 0x66, 64);
	memset(tx_buf + 64 * 6, 0x77, 64);
	memset(tx_buf + 64 * 7, 0x88, 64);
	memset(rx_buf, 0x0, 512);

	uart_stdio_init();
	printf_e("\rSystem start...\n\r");

	ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
	ald_rmu_reset_periperal(RMU_PERH_USB);
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
	ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, ENABLE);
	ald_usb_high_speed_enable(false);
	usb_pin_init();

	usb_hcd_register_driver(0, __host_driver, __nr_host_driver);
	inst = usbh_bulk_driver_open(usr_bulk_cbk);

	usb_mode_set(0, USB_LIB_MODE_NONE, usb_otg_mode_cbk);
	usb_mode_otg_init(0, 100, usbh_pool, USBH_POOL_LEN);

	while (1) {
		usb_mode_otg_main();

		switch (_bulk_host_flag) {
		case STATE_DEVICE_ENUM:
			env.frame  = 0;
			_bulk_host_flag = STATE_DEVICE_READY;
			ret = usbh_bulk_target_speed_get(inst);
			printf_e("\rDevice ready!\n\r");

			if (ret == USB_EP_SPEED_HIGH)
				len = 512;
			else if (ret == USB_EP_SPEED_FULL)
				len = 64;
			else
				len = 8;

			break;

		case  STATE_DEVICE_READY:
			if (env.flag--)
				break;

			env.flag = 600000;
			printf_e("\r------%d-------\n\r", ++env.frame);
			printf_e("\rWrite size#%d\n\r", len);
			memset(tx_buf, 0x50 + env.frame, len);
			ret = usbh_bulk_write(inst, tx_buf, len);
			printf_e("\rRead size#%d\n\r", len);
			memset(rx_buf, 0x0, len);
			ret = usbh_bulk_read(inst, rx_buf, len);
			UNUSED(ret);
			break;

		case STATE_DEVICE_NO:
		case STATE_DEVICE_UNKNOWN:
		case STATE_POWER_FAULT:
			break;

		default:
			break;
		}

		switch (_bulk_dev_flag) {
		case STATE_DEVICE_ENUM:
			printf_e("\rBULK Ready!\n\r");
			_bulk_dev_flag = STATE_DEVICE_READY;
			break;

		case STATE_DEVICE_READ:
			rx_nr = usbd_bulk_rx_packet_avail(&bulk_device);
			printf_e("\rRecv size#%d:\n\r", rx_nr);
			rx_nr = usbd_bulk_packet_read(&bulk_device, rx_buf, rx_nr, 1);

			for (i = 0; i < rx_nr; ++i)
				printf_e("0x%x ", rx_buf[i]);

			printf_e("\r\n");
			printf_e("\rSend size#%d\n\r", rx_nr);
			usbd_bulk_packet_write(&bulk_device, rx_buf, rx_nr, 1);
			_bulk_dev_flag = STATE_DEVICE_READY;
			break;
		case STATE_DEVICE_READY:
		case STATE_DEVICE_NO:
		case STATE_POWER_FAULT:
			break;
		default:
			break;
		}	
	}
}

/**
  * @}
  */
/**
  * @}
  */
