/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"
#include "usbh_hub.h"
#include "usbh_msc.h"
#include "usbh_hid_keyb.h"
#include "usbh_hid_mouse.h"


/** @addtogroup Projects_Examples_USB
  * @{
  */

/** @addtogroup Examples
  * @{
  */

#define STATE_DEVICE_NO		0x01
#define STATE_DEVICE_READY	0x02
#define STATE_DEVICE_INIT_KEYB	0x04
#define STATE_DEVICE_INIT_MOUSE	0x08
#define STATE_DEVICE_INIT_MSC	0x10
#define STATE_DEVICE_UPDATE	0x20
#define STATE_DEVICE_UNKNOWN	0x40
#define STATE_POWER_FAULT	0x80

#define USBH_POOL_LEN	512

uint32_t rx_nr;
env_t env;
uint8_t tx_buf[512];
uint8_t rx_buf0[512];
uint8_t rx_buf1[512];
uint8_t rx_buf2[512];
uint8_t usbh_pool[USBH_POOL_LEN];
uint32_t _hid_flag = STATE_DEVICE_NO;
uint32_t _msc_flag = STATE_DEVICE_NO;
usbh_keyb_t *keyb_inst;
usbh_mouse_t *mouse_inst;
usbh_msc_inst_t *msc_inst;

void usb_hcd_event(void *pdata);
DECLARE_EVENT_DRIVER(usb_event_driver, 0, 0, usb_hcd_event);
static usbh_class_driver_t const * const __host_driver[] = {
	&__hub_class_driver,
	&__usbh_hid_driver,
	&__usbh_msc_driver,
	&usb_event_driver
};
static const uint32_t __nr_host_driver = sizeof(__host_driver) / sizeof(usbh_class_driver_t *);

/**
  * @brief  Initializate pin of USB.
  * @retval None
  */
void usb_pin_init(void)
{
	gpio_init_t x;

	/* Initialize vbus pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_5;
	ald_gpio_init(GPIOB, GPIO_PIN_15, &x);

	return;
}

/**
  * @brief  MSC callback.
  * @param  inst: MSC instance.
  * @param  event: Type of the event.
  * @param  data: message of the event.
  * @retval None
  */
void usr_msc_cbk(usbh_msc_inst_t *inst, uint32_t event, void *data)
{
	switch (event) {
	case MSC_EVENT_OPEN:
		printf_e("\rSTATE_DEVICE_ENUM\n\r");
		_msc_flag = STATE_DEVICE_INIT_MSC;
		break;
	case MSC_EVENT_CLOSE:
		printf_e("\rMSC close\n\r");
		env.flag  = 0;
		_msc_flag = STATE_DEVICE_NO;
		break;
	default:
		break;
	}
}

/**
  * @brief  Mouse callback.
  * @param  inst: Mouse instance.
  * @param  event: Type of the event.
  * @param  param: Parameter.
  * @param  data: message of the event.
  * @retval None
  */
void usr_mouse_cbk(usbh_mouse_t *inst, uint32_t event, uint32_t param, void *data)
{
	switch (event) {
	case USB_EVENT_CONNECTED:
		_hid_flag = STATE_DEVICE_INIT_MOUSE;
		env.update_mouse = 0;
		break;
	case USB_EVENT_DISCONNECTED:
		_hid_flag = STATE_DEVICE_NO;
		env.update_mouse = 0;
		break;
	case USBH_EVENT_HID_MS_PRESS:
		env.button |= param;
		env.update_mouse = 1;
		break;
	case USBH_EVENT_HID_MS_REL:
		env.button &= ~param;
		env.update_mouse = 1;
		break;
	case USBH_EVENT_HID_MS_X:
		env.x_pos += (int8_t)param;
		env.update_mouse = 1;
		break;
	case USBH_EVENT_HID_MS_Y:
		env.y_pos += (int8_t)param;
		env.update_mouse = 1;
		break;
	default:
		env.update_mouse = 0;
		break;
	}
	
	if (env.update_mouse)
		printf_e("Pos:[%d, %d] Botton: 0x%x\n\r", env.x_pos, env.y_pos, env.button);
	
	return;
}

/**
  * @brief  Keyboard callback.
  * @param  inst: Keyboard instance.
  * @param  event: Type of the event.
  * @param  param: Parameter.
  * @param  data: message of the event.
  * @retval None
  */
void usr_keyb_cbk(usbh_keyb_t *inst, uint32_t event, uint32_t param, void *data)
{
	int8_t uchar;
	
	switch (event) {
	case USB_EVENT_CONNECTED:
		_hid_flag = STATE_DEVICE_INIT_KEYB;
		break;
	case USB_EVENT_DISCONNECTED:
		_hid_flag = STATE_DEVICE_NO;
		env.modify = 0;
		break;
	case USBH_EVENT_HID_KB_PRESS:
		if (param == HID_KEYB_USAGE_CAPSLOCK) {
			_hid_flag = STATE_DEVICE_UPDATE;
			env.modify ^= HID_KEYB_CAPS_LOCK;
		}
		else {
			if (param == HID_KEYB_USAGE_BACKSPACE) {
				uchar = 0x08;
			}
			else {
				uchar = usbh_keyb_usage_to_char(keyb_inst, &__keyb_map, param);
				printf_e("--%c--\n\r", uchar);
			}
		}
		break;
	case USBH_EVENT_HID_KB_MOD:
		env.update_keyb = 1;
		printf_e("Keyboard modify\n\r");
		break;
	case USBH_EVENT_HID_KB_REL:
		env.update_keyb = 2;
		printf_e("Keyboard release\n\r");
		break;
	default:
		env.update_keyb = 10;
		break;
	}

	return;
}

/**
  * @brief  Handle HCD event.
  * @param  data: Parameter of the event.
  * @retval None
  */
void usb_hcd_event(void *data)
{
	event_info_t *info = (event_info_t *)data;

	switch (info->event) {
	case USB_EVENT_CONNECTED:
		/* Device is connect */
		printf_e("\rDevice Connected\n\r");
		break;
	case USB_EVENT_DISCONNECTED:
		/* Device is disconnect */
		printf_e("\rDevice Disconnected\n\r");
		_hid_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_POWER_FAULT:
		printf_e("\rSTATE_POWER_FAULT\n\r");
		_hid_flag = STATE_POWER_FAULT;
		break;
	default:
		break;
	}
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
  	uint32_t ret = 0;

	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

	memset(&env, 0x0, sizeof(env_t));
	memset(tx_buf, 0x11, 64);
	memset(tx_buf + 64, 0x22, 64);
	memset(tx_buf + 64 * 2, 0x33, 64);
	memset(tx_buf + 64 * 3, 0x44, 64);
	memset(tx_buf + 64 * 4, 0x55, 64);
	memset(tx_buf + 64 * 5, 0x66, 64);
	memset(tx_buf + 64 * 6, 0x77, 64);
	memset(tx_buf + 64 * 7, 0x88, 64);
	memset(rx_buf0, 0x0, 512);
	memset(rx_buf1, 0x0, 512);

	uart_stdio_init();
	printf_e("\rSystem start...\n\r");

	ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
	ald_rmu_reset_periperal(RMU_PERH_USB);
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
	usb_pin_init();

	usb_hcd_register_driver(0, __host_driver, __nr_host_driver);
	keyb_inst  = usbh_keyb_open(usr_keyb_cbk, rx_buf0, 128);
	mouse_inst = usbh_mouse_open(usr_mouse_cbk, rx_buf1, 128);
	msc_inst   = usbh_msc_driver_open(0, usr_msc_cbk);
	usb_hcd_init(0, usbh_pool, USBH_POOL_LEN);

	while (1) {
		usb_hcd_main();

		switch (_hid_flag) {
		case STATE_DEVICE_INIT_KEYB:
			if ((usbh_keyb_init(keyb_inst)) != 0)
				break;

			printf_e("Keyboard Ready!\n\r");
			_hid_flag = STATE_DEVICE_READY;
			break;

		case STATE_DEVICE_INIT_MOUSE:
			if ((usbh_mouse_init(mouse_inst)) != 0)
				break;

			printf_e("Mouse Ready!\n\r");
			_hid_flag = STATE_DEVICE_READY;
			break;
			
		case  STATE_DEVICE_UPDATE:
			_hid_flag = STATE_DEVICE_READY;
			usbh_keyb_modifier_set(keyb_inst, env.modify);
			break;

		case STATE_DEVICE_READY:
		case STATE_DEVICE_NO:
		case STATE_DEVICE_UNKNOWN:
		case STATE_POWER_FAULT:
			break;
		default:
			break;
		}

		switch (_msc_flag) {
		case STATE_DEVICE_INIT_MSC:
			if ((usbh_msc_driver_ready(msc_inst)) != 0)
				break;

			_msc_flag = STATE_DEVICE_READY;
			break;

		case  STATE_DEVICE_READY:
			if (env.flag)
				break;

			env.flag = 1;
			printf_e("\rDevice ready!\n\r");
			printf_e("\rRead block#2!\n\r");
			ret = usbh_msc_block_read(msc_inst, 2, rx_buf2, 1);
			printf_e("\rWrite block#2!\n\r");
			ret = usbh_msc_block_write(msc_inst, 2, tx_buf, 1);
			printf_e("\rRead block#2!\n\r");
			memset(rx_buf2, 0x0, 512);
			ret = usbh_msc_block_read(msc_inst, 2, rx_buf2, 1);
			UNUSED(ret);
			break;

		default:
			break;
		}
	}
}

/**
  * @}
  */
/**
  * @}
  */
