/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"
#include "usbh_bulk.h"


/** @addtogroup Projects_Examples_USB
  * @{
  */

/** @addtogroup Examples
  * @{
  */

#define STATE_DEVICE_NO		0x01
#define STATE_DEVICE_ENUM	0x02
#define STATE_DEVICE_READY	0x04
#define STATE_DEVICE_UNKNOWN	0x08
#define STATE_POWER_FAULT	0x10

#define USBH_POOL_LEN	128

uint32_t rx_nr;
env_t env;
uint8_t tx_buf[512];
uint8_t rx_buf[512];
uint8_t usbh_pool[USBH_POOL_LEN];
uint32_t _bulk_flag = STATE_DEVICE_NO;
usbh_bulk_inst_t *inst;

void usb_hcd_event(void *pdata);
DECLARE_EVENT_DRIVER(usb_event_driver, 0, 0, usb_hcd_event);
static usbh_class_driver_t const * const __host_driver[] = {
	&__usbh_bulk_driver,
	&usb_event_driver
};
static const uint32_t __nr_host_driver = sizeof(__host_driver) / sizeof(usbh_class_driver_t *);

/**
  * @brief  Initializate pin of USB.
  * @retval None
  */
void usb_pin_init(void)
{
	gpio_init_t x;

	/* Initialize vbus pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_5;
	ald_gpio_init(GPIOB, GPIO_PIN_15, &x);

	return;
}

/**
  * @brief  BULK callback.
  * @param  inst: BULK instance.
  * @param  event: Type of the event.
  * @param  data: message of the event.
  * @retval None
  */
void usr_bulk_cbk(usbh_bulk_inst_t *inst, uint32_t event, void *data)
{
	switch (event) {
	case USBH_BULK_EVENT_OPEN:
		printf_e("\rSTATE_DEVICE_ENUM\n\r");
		_bulk_flag = STATE_DEVICE_ENUM;
		break;
	case USBH_BULK_EVENT_CLOSE:
		printf_e("\rBULK close\n\r");
		_bulk_flag = STATE_DEVICE_NO;
		break;
	default:
		break;
	}
}

/**
  * @brief  Handle HCD event.
  * @param  data: Parameter of the event.
  * @retval None
  */
void usb_hcd_event(void *data)
{
	event_info_t *info = (event_info_t *)data;

	switch (info->event) {
	case USB_EVENT_CONNECTED:
		/* Device is connect */
		printf_e("\rDevice Connected\n\r");
		break;
	case USB_EVENT_DISCONNECTED:
		/* Device is disconnect */
		printf_e("\rDevice Disconnected\n\r");
		env.flag = 0;
		_bulk_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_POWER_FAULT:
		printf_e("\rSTATE_POWER_FAULT\n\r");
		_bulk_flag = STATE_POWER_FAULT;
		break;
	default:
		break;
	}
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
  	uint32_t ret, len;

	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

	memset(&env, 0x0, sizeof(env_t));
	memset(tx_buf, 0x11, 64);
	memset(tx_buf + 64, 0x22, 64);
	memset(tx_buf + 64 * 2, 0x33, 64);
	memset(tx_buf + 64 * 3, 0x44, 64);
	memset(tx_buf + 64 * 4, 0x55, 64);
	memset(tx_buf + 64 * 5, 0x66, 64);
	memset(tx_buf + 64 * 6, 0x77, 64);
	memset(tx_buf + 64 * 7, 0x88, 64);
	memset(rx_buf, 0x0, 512);

	uart_stdio_init();
	printf_e("\rSystem start...\n\r");

	ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
	ald_rmu_reset_periperal(RMU_PERH_USB);
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
	ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, ENABLE);
	usb_pin_init();

	usb_hcd_register_driver(0, __host_driver, __nr_host_driver);
	inst = usbh_bulk_driver_open(usr_bulk_cbk);
	usb_hcd_init(0, usbh_pool, USBH_POOL_LEN);

	while (1) {
		usb_hcd_main();

		switch (_bulk_flag) {
		case STATE_DEVICE_ENUM:
			env.frame  = 0;
			_bulk_flag = STATE_DEVICE_READY;
			ret = usbh_bulk_target_speed_get(inst);
			printf_e("\rDevice ready!\n\r");

			if (ret == USB_EP_SPEED_HIGH)
				len = 512;
			else if (ret == USB_EP_SPEED_FULL)
				len = 64;
			else
				len = 8;

			break;

		case  STATE_DEVICE_READY:
			if (env.flag--)
				break;

			env.flag = 1000000;
			printf_e("\r--------------%d----------------\n\r", ++env.frame);
			printf_e("\rWrite to device, size#%d\n\r", len);
			memset(tx_buf, 0x50 + env.frame, len);
			ret = usbh_bulk_write(inst, tx_buf, len);
			printf_e("\rRead from device, size#%d\n\r", len);
			memset(rx_buf, 0x0, len);
			ret = usbh_bulk_read(inst, rx_buf, len);
			UNUSED(ret);

			if ((memcmp(tx_buf, rx_buf, len)) == 0)
				printf_e("\rData transmission is correct!\n\r");
			else
				printf_e("\rData transmission error!\n\r");

			break;

		case STATE_DEVICE_NO:
		case STATE_DEVICE_UNKNOWN:
		case STATE_POWER_FAULT:
			break;

		default:
			break;
		}
	}
}

/**
  * @}
  */
/**
  * @}
  */
