/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"
#include "usbd_hid_keyb.h"
#include "usbd_hid_mouse.h"
#include "usbd_comp.h"


/** @addtogroup Projects_Examples_USB
  * @{
  */

/** @addtogroup Examples
  * @{
  */

#define STATE_DEVICE_NO		0x01
#define STATE_DEVICE_CONN	0x02
#define STATE_DEVICE_READY	0x04
#define STATE_DEVICE_READ	0x08
#define STATE_DEVICE_WRITE	0x10
#define STATE_DEVICE_IDLE	0x20
#define STATE_POWER_FAULT	0x40
#define NUM_COMP_DEVICE		0x02
#define DESC_BUF_SIZE		0x100

uint32_t rx_nr;
env_t env;
uint8_t tx_buf[512];
uint8_t rx_buf[512];
uint8_t desc_buf[DESC_BUF_SIZE];
uint32_t idx_media = 1;
uint32_t _mouse_flag = STATE_DEVICE_NO;

uint32_t msc_event_handle(void *data, uint32_t event, uint32_t value, void *p_data);
uint32_t mouse_handle(void *data, uint32_t event, uint32_t value, void *p_data);

/**
  * @brief  Initializate pin of USB.
  * @retval None
  */
void usb_pin_init(void)
{
	gpio_init_t x;

	/* Initialize vbus pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_5;
	ald_gpio_init(GPIOB, GPIO_PIN_15, &x);

	return;
}

/**
  * @brief String descriptor
  */
const uint8_t lang_desc[] =
{
	4,
	USB_DTYPE_STRING,
	USBShort(USB_LANG_EN_US)
};

/**
  * @brief Manufact string
  */
const uint8_t manufact_str[] = {
	(17 + 1) * 2,   
	USB_DTYPE_STRING,
	'E', 0, 'a', 0, 's', 0, 't', 0, 's', 0, 'o', 0, 'f', 0, 't', 0, ' ', 0,
	'S', 0, 'h', 0, 'a', 0, 'n', 0, 'g', 0, 'h', 0, 'a', 0, 'i', 0,
};

/**
  * @brief Product string
  */
const uint8_t product_str[] = {
	(13 + 1) * 2,
	USB_DTYPE_STRING,
	'M', 0, 'o', 0, 'u', 0, 's', 0, 'e', 0, ' ', 0, 'E', 0, 'x', 0,
	'a', 0, 'm', 0, 'p', 0, 'l', 0, 'e', 0
};

/**
  * @brief Serial number string
  */
const uint8_t serial_num_str[] =
{
	(8 + 1) * 2,
	USB_DTYPE_STRING,
	'1', 0, '2', 0, '3', 0, '4', 0, '5', 0, '6', 0, '7', 0, '8', 0
};

/**
  * @brief Interface string
  */
const uint8_t data_interface_str[] =
{
	(19 + 1) * 2,
	USB_DTYPE_STRING,
	'H', 0, 'I', 0, 'D', 0, ' ', 0, 'M', 0, 'o', 0, 'u', 0, 's', 0,
	'e', 0, ' ', 0, 'I', 0, 'n', 0, 't', 0, 'e', 0, 'r', 0, 'f', 0,
	'a', 0, 'c', 0, 'e', 0
};

/**
  * @brief Configure string
  */
const uint8_t config_str[] =
{
	(23 + 1) * 2,
	USB_DTYPE_STRING,
	'H', 0, 'I', 0, 'D', 0, ' ', 0, 'M', 0, 'o', 0, 'u', 0, 's', 0,
	'e', 0, ' ', 0, 'C', 0, 'o', 0, 'n', 0, 'f', 0, 'i', 0, 'g', 0,
	'u', 0, 'r', 0, 'a', 0, 't', 0, 'i', 0, 'o', 0, 'n', 0
};

/**
  * @brief String descriptor
  */
const uint8_t *const string_desc[] =
{
	lang_desc,
	manufact_str,
	product_str,
	serial_num_str,
	data_interface_str,
	config_str
};

#define NUM_STRING_DESCRIPTORS (sizeof(string_desc) / sizeof(uint8_t *))

/**
  * @brief  Handle keyboard event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t keyb_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_CONNECTED:
		printf_e("\rCOMP connect!\n\r");
		break;
	case USB_EVENT_DISCONNECTED:
		break;
	case USB_EVENT_TX_COMPLETE:
		break;
	case USB_EVENT_SUSPEND:
		printf_e("\rSuspend!\n\r");
		_mouse_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_RESUME:
		break;
	case USBD_HID_KEYB_EVENT_SET_LEDS:
		printf_e("\rLED Flicker[0x%x]\n\r", value);
		break;
	default:
		break;
	}

	return 0;
}

/**
  * @brief  Handle mouse event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t mouse_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_CONNECTED:
		_mouse_flag = STATE_DEVICE_CONN;
		break;
	case USB_EVENT_DISCONNECTED:
		_mouse_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_TX_COMPLETE:
		break;
	case USB_EVENT_SUSPEND:
		_mouse_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_RESUME:
		break;
	default:
		break;
	}

	return 0;
}

/**
  * @brief Keyboard device information
  */
usbd_hid_keyb_dev_t keyb_device = {
	USB_VID_EASTSOFT_30CC,
	USB_PID_KEYBOARD,
	500,
	USB_CONF_ATTR_SELF_PWR,
	keyb_handle,
	(void *)&keyb_device,
	string_desc,
	NUM_STRING_DESCRIPTORS
};

/**
  * @brief Mouse device information
  */
usbd_hid_mouse_dev_t mouse_device = {
	USB_VID_EASTSOFT_30CC,
	USB_PID_MOUSE,
	500,
	USB_CONF_ATTR_SELF_PWR,
	mouse_handle,
	(void *)&mouse_device,
	string_desc,
	NUM_STRING_DESCRIPTORS
};

/**
  * @brief Composite entry array
  */
comp_entry_t entry[NUM_COMP_DEVICE];

/**
  * @brief Composite device information
  */
usbd_comp_dev_t comp_device =
{
	USB_VID_EASTSOFT_30CC,
	USB_PID_COMP_HID_HID,
	500,
	USB_CONF_ATTR_SELF_PWR,
	0,
	string_desc,
	NUM_STRING_DESCRIPTORS,
	NUM_COMP_DEVICE,
	entry,
};

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{	
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);

	memset(&env, 0x0, sizeof(env_t));
	memset(tx_buf, 0x0, 128);
	memset(rx_buf, 0x0, 128);

	uart_stdio_init();
	printf_e("System start...\n\r");

	ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
	ald_rmu_reset_periperal(RMU_PERH_USB);
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
	ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, ENABLE);
	usb_pin_init();
	ald_usb_high_speed_enable(false);
	ald_usb_otg_session_request(true);

	comp_device.device[0].inst = usbd_hid_keyb_init_comp(0, &keyb_device, &entry[0]);
	comp_device.device[1].inst = usbd_hid_mouse_init_comp(0, &mouse_device, &entry[1]);
	usbd_comp_init(0, &comp_device, DESC_BUF_SIZE, desc_buf);

	while (1) {
		switch (_mouse_flag) {
		case STATE_DEVICE_CONN:
			printf_e("\rCOMP Ready!\n\r");
			_mouse_flag = STATE_DEVICE_READY;
			break;
		case  STATE_DEVICE_READY:
			ald_delay_ms(5000);
			printf_e("\rMove Increase\n\r");
			usbd_hid_mouse_state_change(&mouse_device, 5, 5, 0x0);
			ald_delay_ms(500);
			printf_e("\rCAPSLOCK!\n\r");
			usbd_hid_keyb_state_change(&keyb_device, 0, HID_KEYB_USAGE_CAPSLOCK, 1);
			ald_delay_ms(50);
			usbd_hid_keyb_state_change(&keyb_device, 0, HID_KEYB_USAGE_CAPSLOCK, 0);
			ald_delay_ms(3000);
			printf_e("\rT!\n\r");
			usbd_hid_keyb_state_change(&keyb_device, 0, HID_KEYB_USAGE_T, 1);
			ald_delay_ms(50);
			usbd_hid_keyb_state_change(&keyb_device, 0, HID_KEYB_USAGE_T, 0);
			break;
		case STATE_DEVICE_NO:
		case STATE_POWER_FAULT:
			break;
		default:
			break;
		}
	}
}

/**
  * @}
  */
/**
  * @}
  */
