/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"
#include "usbd_audio.h"


/** @addtogroup Projects_Examples_USB
  * @{
  */

/** @addtogroup Examples
  * @{
  */

#define STATE_DEVICE_NO		0x01
#define STATE_DEVICE_CONN	0x02
#define STATE_DEVICE_READY	0x04
#define STATE_DEVICE_READ	0x08
#define STATE_DEVICE_WRITE	0x10
#define STATE_DEVICE_IDLE	0x20
#define STATE_POWER_FAULT	0x40

uint32_t rx_nr;
env_t env;
uint8_t tx_buf[512];
uint8_t rx_buf[512];
uint32_t _audio_flag = STATE_DEVICE_NO;

void data_handle(void *data, uint32_t param, uint32_t event);
uint32_t audio_handle(void *data, uint32_t event, uint32_t value, void *p_data);


/**
  * @brief  Initializate pin of USB.
  * @retval None
  */
void usb_pin_init(void)
{
	gpio_init_t x;

	/* Initialize vbus pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_5;
	ald_gpio_init(GPIOB, GPIO_PIN_15, &x);

	return;
}

/**
  * @brief String descriptor
  */
const uint8_t lang_desc[] =
{
	4,
	USB_DTYPE_STRING,
	USBShort(USB_LANG_EN_US)
};

/**
  * @brief Manufact string
  */
const uint8_t manufact_str[] = {
	(17 + 1) * 2,   
	USB_DTYPE_STRING,
	'E', 0, 'a', 0, 's', 0, 't', 0, 's', 0, 'o', 0, 'f', 0, 't', 0,
	' ', 0, 'S', 0, 'h', 0, 'a', 0, 'n', 0, 'g', 0, 'h', 0, 'a', 0,
	'i', 0,
};

/**
  * @brief Product string
  */
const uint8_t product_str[] = {
	(20 + 1) * 2, 
	USB_DTYPE_STRING, 
	'G', 0, 'e', 0, 'n', 0, 'e', 0, 'r', 0, 'i', 0, 'c', 0, ' ', 0,
	'A', 0, 'u', 0, 'd', 0, 'i', 0, 'o', 0, ' ', 0, 'D', 0, 'e', 0,
	'v', 0, 'i', 0, 'c', 0, 'e', 0
};

/**
  * @brief Serial number string
  */
const uint8_t serial_num_str[] =
{
	(8 + 1) * 2,
	USB_DTYPE_STRING,
	'1', 0, '2', 0, '3', 0, '4', 0, '5', 0, '6', 0, '7', 0, '8', 0
};

/**
  * @brief Interface string
  */
const uint8_t data_interface_str[] =
{
	(20 + 1) * 2,
	USB_DTYPE_STRING,
	'A', 0, 'u', 0, 'd', 0, 'i', 0, 'o', 0, ' ', 0, 'D', 0, 'a', 0,
	't', 0, 'a', 0, ' ', 0, 'I', 0, 'n', 0, 't', 0, 'e', 0, 'r', 0,
	'f', 0, 'a', 0, 'c', 0, 'e', 0
};

/**
  * @brief Configure string
  */
const uint8_t config_str[] =
{
	(24 + 1) * 2,
	USB_DTYPE_STRING,
	'A', 0, 'u', 0, 'd', 0, 'i', 0, 'o', 0, ' ', 0, 'D', 0, 'a', 0,
	't', 0, 'a', 0, ' ', 0, 'C', 0, 'o', 0, 'n', 0, 'f', 0, 'i', 0,
	'g', 0, 'u', 0, 'r', 0, 'a', 0, 't', 0, 'i', 0, 'o', 0, 'n', 0
};

/**
  * @brief String descriptor
  */
const uint8_t *const string_desc[] =
{
	lang_desc,
	manufact_str,
	product_str,
	serial_num_str,
	data_interface_str,
	config_str
};

#define NUM_STRING_DESCRIPTORS (sizeof(string_desc) / sizeof(uint8_t *))

/**
  * @brief Audio device information
  */
usbd_audio_dev_t audio_device = {
	USB_VID_EASTSOFT_30CC,
	USB_PID_AUDIO,
	"EASTSOFT",
	"Audio Storage   ",
	"1.00",
	500,
	USB_CONF_ATTR_SELF_PWR,
	audio_handle,
	string_desc,
	NUM_STRING_DESCRIPTORS,
	60,
	0,
	1,
};

/**
  * @brief  Handle audio event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t audio_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_CONNECTED:
		_audio_flag = STATE_DEVICE_CONN;
		break;
	case USB_EVENT_DISCONNECTED:
		break;
	case USB_EVENT_SUSPEND:
		printf_e("\rSuspend!\n\r");
		_audio_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_RESUME:
		break;

	default:
		break;
	}

	return 0;
}

/**
  * @brief  Handle data.
  * @param  data: Parameter of the event.
  * @param  param: Paramter.
  * @param  event: Type of the event.
  * @retval None
  */
void data_handle(void *data, uint32_t param, uint32_t event)
{
	switch (event) {
	case USBD_AUDIO_EVENT_DATAOUT:
		printf_e("\rRecv#%d\n\r", env.cnt++);
		break;
	default:
		break;
	}

	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);

	memset(&env, 0x0, sizeof(env_t));
	memset(tx_buf, 0x0, 128);
	memset(rx_buf, 0x0, 128);

	uart_stdio_init();
	printf_e("\rSystem start...\n\r");

	ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
	ald_rmu_reset_periperal(RMU_PERH_USB);
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
	ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, ENABLE);
	usb_pin_init();
	ald_usb_high_speed_enable(false);
	ald_usb_otg_session_request(true);
	usbd_audio_init(0, &audio_device);

	while (1) {
		switch (_audio_flag) {
		case STATE_DEVICE_CONN:
			printf_e("\rAudio Ready!\n\r");
			usbd_audio_data_out(&audio_device, rx_buf, 192, data_handle);
			_audio_flag = STATE_DEVICE_READY;
			break;

		case STATE_DEVICE_READY:
		case STATE_DEVICE_NO:
		case STATE_POWER_FAULT:
			break;
		default:
			break;
		}
	}
}

/**
  * @}
  */
/**
  * @}
  */
