/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Board Initialization
  *
  * @version V1.0
  * @date    16 Sep 2019
  * @author  AE Team
  * @note   
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */ 

#include <rthw.h>
#include <rtthread.h>
#include <string.h>
#include "ald_cmu.h"
#include "ald_gpio.h"
#include "ald_uart.h"
#include "board.h"


uart_handle_t h_uart;
struct rt_device __uart1;
rt_err_t uart1_init(rt_device_t dev);
rt_err_t uart1_open(rt_device_t dev, rt_uint16_t oflag);
rt_size_t uart1_read(rt_device_t dev, rt_off_t pos, void *buffer, rt_size_t size);
rt_size_t uart1_write(rt_device_t dev, rt_off_t pos, const void *buffer, rt_size_t size);

/**
 * This function will initial STM32 board.
 */
void rt_hw_board_init(void)
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_96M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 96000000);

	/* Enable all peripherals */
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	/* Set SysTick as 10ms */
	ald_systick_interval_select(SYSTICK_INTERVAL_10MS);
	
	memset(&__uart1, 0x0, sizeof(struct rt_device));

	__uart1.init  = uart1_init;
	__uart1.open  = uart1_open;
	__uart1.read  = uart1_read;
	__uart1.write = uart1_write;
	rt_device_register(&__uart1, "uart1", RT_DEVICE_FLAG_RDWR);

#ifdef RT_USING_HEAP
	rt_system_heap_init((void *)HEAP_BEGIN, (void *)HEAP_END);
#endif
#ifdef RT_USING_COMPONENTS_INIT
	rt_components_board_init();
#endif
#ifdef RT_USING_CONSOLE
	rt_console_set_device(RT_CONSOLE_DEVICE_NAME);
#endif
}

/**
 * This is the systick interrupt service routine.
 *
 */
void rt_tick_handler(void)
{
	/* enter interrupt */
	rt_interrupt_enter();
	rt_tick_increase();
	/* leave interrupt */
	rt_interrupt_leave();
}

/**
  * @brief  Initialize the UART1
  * @retval None
  */
rt_err_t uart1_init(rt_device_t dev)
{
	gpio_init_t x;

	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_UART1, ENABLE);
	
	/* Initialize tx pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_0, &x);

	/* Initialize rx pin */
	x.mode  = GPIO_MODE_INPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_1, &x);

	/* Initialize uart */
	h_uart.perh             = UART1;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);

	return 0;
}

rt_err_t uart1_open(rt_device_t dev, rt_uint16_t oflag)
{
	return uart1_init(dev);
}

rt_size_t uart1_read(rt_device_t dev, rt_off_t pos, void *buffer, rt_size_t size)
{
	ald_uart_recv_n_lock(&h_uart, (uint8_t *)buffer, size, 10);
	return size;
}


rt_size_t uart1_write(rt_device_t dev, rt_off_t pos, const void *buffer, rt_size_t size)
{
	ald_uart_send_n_lock(&h_uart, (uint8_t *)buffer, size, 1000);
	return size;
}



