/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Nov 2019
  * @author  AE Team
  * @note   
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */ 

#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
  
lv_obj_t * led1;
lv_obj_t * led2;
lv_obj_t * led3;
lv_obj_t * btn1;
lv_obj_t * slider1;
lv_obj_t * gauge1;

void lv_ex_led(void);
void lv_ex_button(void);
void lv_ex_gauge_1(void);
void lv_ex_slider_1(void);
static void event_handler(lv_obj_t * obj, lv_event_t event);

/**
  * @brief  Button pressed callback function
  * @param  btn: pointer to button object 
  * @param  event: button event
  * @retval None
  */	
void btn_event_cb(lv_obj_t * btn, lv_event_t event)
{
	switch(event) {
		case LV_EVENT_PRESSED:
		lv_led_on(led1);	
		break;
		case LV_EVENT_SHORT_CLICKED:
		lv_led_on(led2);	
		break;
		case LV_EVENT_CLICKED:
		lv_led_on(led3);	
		break;
		case LV_EVENT_LONG_PRESSED:
		lv_led_off(led1);	
		break;
		case LV_EVENT_LONG_PRESSED_REPEAT:
		lv_led_toggle(led3);	
		break;
		case LV_EVENT_RELEASED:
			
		break;
	}
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	lv_init();
	lv_port_disp_init();
	lv_port_indev_init();
	lv_ex_button();
	lv_ex_led();
	lv_ex_gauge_1();
	lv_ex_slider_1();
	while(1) {
		ald_delay_ms(10);
		lv_task_handler();
	}
}

/**
  * @brief  Button display configuration
  * @retval None
  */
void lv_ex_button(void)
{
	btn1 = lv_btn_create(lv_scr_act(), NULL); /*Add a button the current screen*/
	lv_obj_align(btn1, NULL, LV_ALIGN_CENTER,0, 0);
	lv_obj_set_size(btn1, 100, 50); /*Set its size*/
	lv_obj_set_event_cb(btn1, btn_event_cb); /*Assign a callback to the button*/
	lv_obj_t * label = lv_label_create(btn1, NULL); /*Add a label to the button*/
	lv_label_set_text(label, "Button1"); /*Set the labels text*/
}

/**
  * @brief  Led display configuration
  * @retval None
  */
void lv_ex_led(void)
{
	/*Create a style for the LED*/
	static lv_style_t style_led;
	
	lv_style_copy(&style_led, &lv_style_pretty_color);
	style_led.body.radius = LV_RADIUS_CIRCLE;
	style_led.body.main_color = LV_COLOR_MAKE(0xb5, 0x0f, 0x04);
	style_led.body.grad_color = LV_COLOR_MAKE(0x50, 0x07, 0x02);
	style_led.body.border.color = LV_COLOR_MAKE(0xfa, 0x0f, 0x00);
	style_led.body.border.width = 3;
	style_led.body.border.opa = LV_OPA_30;
	style_led.body.shadow.color = LV_COLOR_MAKE(0xb5, 0x0f, 0x04);
	style_led.body.shadow.width = 5;
	/*Create a LED and switch it ON*/
	 led1 = lv_led_create(lv_scr_act(), NULL);
	lv_obj_set_style(led1, &style_led);
	lv_obj_align(led1, NULL, LV_ALIGN_CENTER, -80, -100);
	lv_led_off(led1);
	/*Copy the previous LED and set a brightness*/
	 led2 = lv_led_create(lv_scr_act(), led1);
	lv_obj_align(led2, NULL, LV_ALIGN_CENTER, 0, -100);
	lv_led_set_bright(led2, 0xff);//190);
	/*Copy the previous LED and switch it OFF*/
	 led3 = lv_led_create(lv_scr_act(), led1);
	lv_obj_align(led3, NULL, LV_ALIGN_CENTER, 80, -100);
	lv_led_on(led3);
}

/**
  * @brief  Gauge display configuration
  * @retval None
  */
void lv_ex_gauge_1(void)
{
	/*Create a style*/
	static lv_style_t style;
	
	lv_style_copy(&style, &lv_style_pretty_color);
	style.body.main_color = lv_color_hex3(0x666); /*Line color at the beginning*/
	style.body.grad_color = lv_color_hex3(0x666); /*Line color at the end*/
	style.body.padding.left = 10; /*Scale line length*/
	style.body.padding.inner = 8 ; /*Scale label padding*/
	style.body.border.color = lv_color_hex3(0x333); /*Needle middle circle color*/
	style.line.width = 3;
	style.text.color = lv_color_hex3(0x333);
	style.line.color = LV_COLOR_RED; /*Line color after the critical?
	,!value*/
	/*Describe the color for the needles {0x001F, 0xFD20, 0x8010} */
	static lv_color_t needle_colors[3] = {0x001F, 0xFD20, 0x8010};//{LV_COLOR_BLUE, LV_COLOR_ORANGE, LV_COLOR_PURPLE};
	/*Create a gauge*/
	gauge1 = lv_gauge_create(lv_scr_act(), NULL);
	lv_gauge_set_style(gauge1, LV_GAUGE_STYLE_MAIN, &style);
	lv_gauge_set_needle_count(gauge1, 1, needle_colors);
	lv_obj_set_size(gauge1, 200, 200);
	lv_obj_align(gauge1, NULL, LV_ALIGN_CENTER, 0, 200);
	/*Set the values*/
	lv_gauge_set_value(gauge1, 0, 0);
}

/**
  * @brief  Slider display configuration
  * @retval None
  */
void lv_ex_slider_1(void)
{
	/*Create styles*/
	static lv_style_t style_bg;
	static lv_style_t style_indic;
	static lv_style_t style_knob;
	
	lv_style_copy(&style_bg, &lv_style_pretty);
	style_bg.body.main_color = LV_COLOR_BLACK;
	style_bg.body.grad_color = LV_COLOR_GRAY;
	style_bg.body.radius = LV_RADIUS_CIRCLE;
	style_bg.body.border.color = LV_COLOR_WHITE;
	lv_style_copy(&style_indic, &lv_style_pretty_color);
	style_indic.body.radius = LV_RADIUS_CIRCLE;
	style_indic.body.shadow.width = 8;
	style_indic.body.shadow.color = style_indic.body.main_color;
	style_indic.body.padding.left = 3;
	style_indic.body.padding.right = 3;
	style_indic.body.padding.top = 3;
	style_indic.body.padding.bottom = 3;
	lv_style_copy(&style_knob, &lv_style_pretty);
	style_knob.body.radius = LV_RADIUS_CIRCLE;
	style_knob.body.opa = LV_OPA_70;
	style_knob.body.padding.top = 10 ;
	style_knob.body.padding.bottom = 10 ;
	/*Create a slider*/
	slider1 = lv_slider_create(lv_scr_act(), NULL);
	lv_slider_set_style(slider1, LV_SLIDER_STYLE_BG, &style_bg);
	lv_slider_set_style(slider1, LV_SLIDER_STYLE_INDIC,&style_indic);
	lv_slider_set_style(slider1, LV_SLIDER_STYLE_KNOB, &style_knob);
	lv_obj_align(slider1, NULL, LV_ALIGN_CENTER, 0, 60);
	lv_obj_set_event_cb(slider1, event_handler);
	
	lv_slider_set_range(slider1, 0, 100);
}

/**
  * @brief  Slider pressed callback function
  * @param  obj: pointer to object 
  * @param  event: slider event
  * @retval None
  */
static void event_handler(lv_obj_t * obj, lv_event_t event)
{
	int16_t tmp = 0;
	
	if(event == LV_EVENT_VALUE_CHANGED) {
		tmp = lv_slider_get_value(slider1);
		lv_gauge_set_value(gauge1,0,tmp);
	}
}

/**
  * @brief  Systick callback function
  * @retval None
  */
void ald_systick_irq_cbk(void)
{
	lv_tick_inc(1);
}

/**
  * @}
  */
/**
  * @}
  */
