/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Example DEMO
  *
  * @version V1.0
  * @date    17 Dec 2020
  * @author  AE Team
  * @note   
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */ 
#include "main.h"


/* Macro definition of eventgroup*/
#define EVENTGROUP_TIMEOUT      10000
#define APP1_WAIT_BITS          (BIT(1)|BIT(3))
#define APP2_WAIT_BITS          (BIT(2)|BIT(4))

/* Task functions */
static void vtask_app1(void *pvParameters);
static void vtask_app2(void *pvParameters);
static void vtask_key_scan(void *pvParameters);
static void vtask_info_print(void *pvParameters);
static void app_task_create (void);

/* This variable is eventgroup handle */
static EventGroupHandle_t event_group;

/**
  * @brief  This is main function.
  * @retval Status
  */
int main(void)
{
	ald_cmu_init();
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	ald_systick_interval_select(SYSTICK_INTERVAL_10MS);

	NVIC_SetPriorityGrouping(NVIC_PRIORITY_GROUP_4);

	bsp_led_init();
	bsp_key_init();
	bsp_uart_init();
	        
	app_task_create();
        
	vTaskStartScheduler();
        
	while(1);
}

/**
  * @brief  task create function
  * @param  None
  * @retval None
  */
static void app_task_create (void)
{
	xTaskCreate( vtask_app1,       
                     "vtask_app1",         
                     512,                 
                     NULL,                
                     1,                   
                     NULL ); 
	
	xTaskCreate( vtask_app2,     
                     "vtask_app2",   
                     512,             
                     NULL,            
                     2,              
                     NULL );
        
	xTaskCreate( vtask_info_print,     
                     "vtask_info_print",   
                     256,            
                     NULL,           
                     3,              
                     NULL ); 
        
	xTaskCreate( vtask_key_scan,      /* task function  */
                     "vtask_key_scan",    /* task name */
                     256,                 /* size of stack, unit is Word */
                     NULL,                /* task parameter */
                     4,                   /* task priority */
                     NULL );              /* task handler */		 
}

/**
  * @brief  information print task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_info_print(void *pvParameters)
{
	uint8_t uccount = 0;
	uint8_t uckey_code;
        
	while(1) {
		uckey_code = bsp_get_key();
		
		if (uckey_code != KEY_NONE) {
			switch (uckey_code) {
				case KEY_DOWN_CENTER:
					printf("Center key pressed down. \r\n");
				break;
				case KEY_DOWN_TOP:
					xEventGroupSetBits(event_group, BIT(2));
					printf("Set eventgroup bit2.  \r\n");
				break;
				case KEY_DOWN_BOTTOM:
					xEventGroupSetBits(event_group, BIT(4));
					printf("Set eventgroup bit4.  \r\n");
				break;
				case KEY_DOWN_LEFT:
					xEventGroupSetBits(event_group, BIT(1));
					printf("Set eventgroup bit1.  \r\n");
				break;
				case KEY_DOWN_RIGHT:
					xEventGroupSetBits(event_group, BIT(3));
					printf("Set eventgroup bit3.  \r\n");
				break;
								
				default:                     
				break;
			}
		}
		vTaskDelay(pdMS_TO_TICKS(10));
	}
}

/**
  * @brief  app1 task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_app1(void *pvParameters)
{
	EventBits_t event_bits;

	if (NULL == event_group) {
		while(1) {
			vTaskDelay(pdMS_TO_TICKS(500));
			bsp_led_toggle(1);
		}
	}

	while(1) {
		event_bits = xEventGroupWaitBits(event_group, APP1_WAIT_BITS, pdFALSE, pdTRUE, pdMS_TO_TICKS(EVENTGROUP_TIMEOUT));
		if ((event_bits&APP1_WAIT_BITS) == APP1_WAIT_BITS) {
			bsp_led_toggle(1);
			printf("App1 xEventGroupWaitBits() returned because APP1_WAIT_BITS both bits were set.  \r\n");
			xEventGroupClearBits(event_group, APP1_WAIT_BITS);
		}
		else {
			printf("App1 xEventGroupWaitBits() returned because xTicksToWait ticks passed.  \r\n");
		}
	}        
}

/**
  * @brief  app2 task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_app2(void *pvParameters)
{
	EventBits_t event_bits;
        
	if (NULL == event_group) {
		while(1) {
			vTaskDelay(pdMS_TO_TICKS(500));
			bsp_led_toggle(2);
		}
	}        

	while(1) {
		event_bits = xEventGroupWaitBits(event_group, APP2_WAIT_BITS, pdTRUE, pdTRUE, pdMS_TO_TICKS(EVENTGROUP_TIMEOUT));
		if ((event_bits&APP2_WAIT_BITS) == APP2_WAIT_BITS) {
			bsp_led_toggle(2);
			printf("App2 xEventGroupWaitBits() returned because APP2_WAIT_BITS both bits were set.  \r\n");
			xEventGroupClearBits(event_group, APP2_WAIT_BITS);
		}
		else {
			printf("App2 xEventGroupWaitBits() returned because xTicksToWait ticks passed.  \r\n");
		}                
	}    
}

/**
  * @brief  key scan task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_key_scan(void *pvParameters)
{                            
	printf("System start...\r\n");

	event_group = xEventGroupCreate();
	if (NULL == event_group) {
		printf("The event group was not created because there was insufficient FreeRTOS heap available.  \r\n");
	}
	else {
		printf("The event group was created.  \r\n");
	}
        
	while(1) {
		bsp_key_scan();
		vTaskDelay(pdMS_TO_TICKS(10));
	}
}
