/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Example DEMO
  *
  * @version V1.0
  * @date    17 Dec 2020
  * @author  AE Team
  * @note   
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */ 
#include "main.h"

/* Task functions */
static void vtask_user_if(void *pvParameters);
static void vtask_led(void *pvParameters);
static void vtask_msg_pro(void *pvParameters);
static void vtask_start(void *pvParameters);
static void app_task_create (void);

/* Task handlers */
static TaskHandle_t xhandle_task_msg_pro = NULL;

/**
  * @brief  This is main function.
  * @retval Status
  */
int main(void)
{
	/* Initialize ALD */
	ald_cmu_init();
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_72M);	
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	ald_systick_interval_select(SYSTICK_INTERVAL_10MS);

	NVIC_SetPriorityGrouping(NVIC_PRIORITY_GROUP_4);

	bsp_led_init();
	bsp_key_init();
	bsp_uart_init();

	app_task_create();
	vTaskStartScheduler();

	while(1);	
}

/**
  * @brief  key task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_user_if(void *pvParameters)
{
	uint8_t uccount = 0;
	uint8_t uckey_code;

	while(1) {
		uckey_code = bsp_get_key();
	
		if (uckey_code != KEY_NONE) {
			switch (uckey_code) {
			case KEY_DOWN_CENTER:
				printf("Center key pressed down. \r\n");
				xTaskNotify(xhandle_task_msg_pro,      	 /* task being notified */
						uccount++,               /* notification value */
						eSetValueWithOverwrite); /* unconditionally set to the value of uccount */       
				break;
			case KEY_DOWN_TOP:
				printf("TOP key pressed down. \r\n");
				break;
			case KEY_DOWN_BOTTOM:
				printf("BOTTOM key pressed down. \r\n");
				break;
			case KEY_DOWN_LEFT:
				printf("LEFT key pressed down. \r\n");
				break;
			case KEY_DOWN_RIGHT:
				printf("RIGHT key pressed down. \r\n");
				break;

			default:                     
				break;
			}
		}
		vTaskDelay(pdMS_TO_TICKS(10));		
	}
}

/**
  * @brief  led task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_led(void *pvParameters)
{
	TickType_t xlast_wake_time;
	const TickType_t xfreq = 200;
	
	xlast_wake_time = xTaskGetTickCount();

	while(1) {
		bsp_led_toggle(2);
		vTaskDelayUntil(&xlast_wake_time, pdMS_TO_TICKS(xfreq));
	}
}

/**
  * @brief  start task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_start(void *pvParameters)
{
	while(1) {
		bsp_key_scan();
		vTaskDelay(pdMS_TO_TICKS(10));
	}
}

/**
  * @brief  process task entry function
  * @param  parameter: user's paramter
  * @retval None
  */
static void vtask_msg_pro(void *pvParameters)
{
	BaseType_t xresult;
	uint32_t ulnotified_value;

	while(1) {
		xresult = xTaskNotifyWait(0x00,      		
				          0xffffffff,       	
			                  &ulnotified_value, 	
				          portMAX_DELAY);   	
		if(xresult == pdPASS) {
			bsp_led_toggle(1);
		}
	}
}

/**
  * @brief  task create function
  * @param  parameter: user's paramter
  * @retval None
  */
static void app_task_create (void)
{
	xTaskCreate( vtask_user_if,       /* task function  */
                     "vtask_sser_if",     /* task name */
                     512,                 /* size of stack, unit is Word */
                     NULL,                /* task parameter */
                     1,                   /* task priority */
                     NULL );              /* task handler */

	xTaskCreate( vtask_led,       
                     "vtask_led",         
                     256,                 
                     NULL,                
                     2,                   
                     NULL ); 

	xTaskCreate( vtask_msg_pro,     
                     "vtask_msg_pro",   
                     512,             
                     NULL,            
                     3,              
                     &xhandle_task_msg_pro );  

	xTaskCreate( vtask_start,     
                     "vtask_start",   
                     512,            
                     NULL,           
                     4,              
                     NULL );         
}
