/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"
#include "ald_gpio.h"
#include "ald_cmu.h"
#include "ald_pmu.h"
#include "ald_rmu.h"
#include "ald_usb.h"
#include "ald_crc.h"
#include "usbd_hid_custom.h"
#include "ald_iap.h"
#include "es_hid_lab.h"
#include "app_update.h"

/** @addtogroup Projects_Examples_USB
  * @{
  */

/** @addtogroup Examples
  * @{
  */

#define STATE_DEVICE_NO		0x01
#define STATE_DEVICE_CONN	0x02
#define STATE_DEVICE_READY	0x04
#define STATE_DEVICE_READ	0x08
#define STATE_DEVICE_WRITE	0x10
#define STATE_DEVICE_IDLE	0x20
#define STATE_POWER_FAULT	0x40

uint32_t tmp;
env_t env;
uint8_t tx_buf[CUSTOMHID_REPORT_SIZE];
uint8_t rx_buf[CUSTOMHID_REPORT_SIZE];
uint32_t idx_media = 1;
uint32_t _hid_boot_flag = STATE_DEVICE_NO;

uint32_t hid_boot_handle(void *data, uint32_t event, uint32_t value, void *p_data);

/**
  * @brief  Initializate pin of USB.
  * @retval None
  */
void usb_pin_init(void)
{
	gpio_init_t x;

	/* Initialize vbus pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_5;
	ald_gpio_init(GPIOB, GPIO_PIN_15, &x);

	return;
}

/**
  * @brief String descriptor
  */
const uint8_t lang_desc[] =
{
	4,
	USB_DTYPE_STRING,
	USBShort(USB_LANG_EN_US)
};

/**
  * @brief Manufact string
  */
const uint8_t manufact_str[] = {
	(17 + 1) * 2,   
	USB_DTYPE_STRING,
	'E', 0, 'a', 0, 's', 0, 't', 0, 's', 0, 'o', 0, 'f', 0, 't', 0, ' ', 0,
	'S', 0, 'h', 0, 'a', 0, 'n', 0, 'g', 0, 'h', 0, 'a', 0, 'i', 0,
};

/**
  * @brief Product string
  */
const uint8_t product_str[] = {
	(25 + 1) * 2,
	USB_DTYPE_STRING,
	'E', 0, 'S', 0, '3', 0, '2', 0, ' ', 0, 'H', 0, 'i', 0, 'd', 0, ' ', 0, 
	'B', 0, 'o', 0, 'o', 0, 't', 0, 'l', 0, 'o', 0, 'd', 0, 'e', 0, 'r', 0, 
	' ', 0, 'D', 0, 'e', 0, 'v', 0, 'i', 0, 'c', 0, 'e', 0,
};

/**
  * @brief Serial number string
  */
const uint8_t serial_num_str[] =
{
	(8 + 1) * 2,
	USB_DTYPE_STRING,
	'1', 0, '2', 0, '3', 0, '4', 0, '5', 0, '6', 0, '7', 0, '8', 0
};

/**
  * @brief Interface string
  */
const uint8_t data_interface_str[] =
{
	(29 + 1) * 2,
	USB_DTYPE_STRING,
	'H', 0, 'i', 0, 'd', 0, ' ', 0, 'U', 0, 's', 0, 'e', 0, 'r', 0, 'S', 0, 
	'p', 0, 'e', 0, 'c', 0, ' ', 0, 'D', 0, 'e', 0, 'v', 0, 'i', 0, 'c', 0,
	'e', 0, ' ', 0, 'I', 0, 'n', 0, 't', 0, 'e', 0, 'r', 0, 'f', 0, 'a', 0,
	'c', 0, 'e', 0 
};

/**
  * @brief Configure string
  */
const uint8_t config_str[] =
{
	(32 + 1) * 2,
	USB_DTYPE_STRING,
	'H', 0, 'i', 0, 'd', 0, ' ', 0, 'U', 0, 's', 0, 'e', 0, 'r', 0, 'S', 0, 
	'p', 0, 'e', 0, 'c', 0, ' ', 0, 'D', 0, 'e', 0, 'v', 0, 'i', 0, 'c', 0,
	'e', 0, ' ', 0, 'C', 0, 'o', 0, 'n', 0, 'f', 0, 'i', 0, 'g', 0, 'u', 0,
	'r', 0, 'a', 0, 't', 0, 'i', 0, 'o', 0, 'n', 0
};

/**
  * @brief String descriptor
  */
const uint8_t *const string_desc[] =
{
	lang_desc,
	manufact_str,
	product_str,
	serial_num_str,
	data_interface_str,
	config_str
};

#define NUM_STRING_DESCRIPTORS (sizeof(string_desc) / sizeof(uint8_t *))

/**
  * @brief Keyboard device information
  */
usbd_hid_boot_dev_t hid_boot_device = {
	USB_VID_EASTSOFT_30CC,
	USB_PID_GAMEPAD,
	500,
	USB_CONF_ATTR_SELF_PWR,
	hid_boot_handle,
	(void *)&hid_boot_device,
	string_desc,
	NUM_STRING_DESCRIPTORS
};

/**
  * @brief  Handle keyboard event.
  * @param  data: Parameter of the event.
  * @param  event: Type of the event.
  * @param  value: Value of the event.
  * @param  p_data: Message of the event.
  * @retval Status.
  */
uint32_t hid_boot_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_CONNECTED:
		printf_e("\rConnect!\n\r");
		_hid_boot_flag = STATE_DEVICE_CONN;
		break;
	case USB_EVENT_DISCONNECTED:
		break;
	case USB_EVENT_RX_AVAILABLE:
		usbd_hid_boot_report_recv(&hid_boot_device, rx_buf, CUSTOMHID_REPORT_SIZE);
		memset(tx_buf, 0, CUSTOMHID_REPORT_SIZE);
		es_lab_commands_process(rx_buf, tx_buf);
		usbd_hid_boot_report_send(&hid_boot_device, tx_buf, CUSTOMHID_REPORT_SIZE);
		break;
	case USB_EVENT_TX_COMPLETE:
		break;
	case USB_EVENT_SUSPEND:
		printf_e("\rSuspend!\n\r");
		_hid_boot_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_RESUME:
		break;
	default:
		break;
	}

	return 0;
}

/**
  * @brief  Check app CRC
  * @param  page: total page of flash data.
  * @retval 0-success, other value indicates failed.
  */
uint32_t check_app_crc(uint8_t *src, uint32_t len)
{
    crc_handle_t h_crc;
   
    /* Initialize CRC */
    h_crc.perh          = CRC;
    h_crc.init.mode     = CRC_MODE_32;
    h_crc.init.seed     = 0xFFFFFFFF;
    h_crc.init.data_rev = ENABLE;
    h_crc.init.data_inv = DISABLE;
    h_crc.init.chs_rev  = ENABLE;
    h_crc.init.chs_inv  = ENABLE;
    ald_crc_init(&h_crc);
    
    return ald_crc_calculate(&h_crc, (uint8_t *)src, len);
}

extern uint32_t actual_size;
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	volatile uint32_t app_crc = 0;
	volatile uint32_t calc_crc = 0;
	volatile uint32_t app_size = 0;
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	
	memset(&env, 0x0, sizeof(env_t));
	memset(tx_buf, 0x0, 128);
	memset(rx_buf, 0x0, 128);

	uart_stdio_init();
	printf_e("\rSystem start...\n\r");

	ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_CRC, ENABLE);
	ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
	ald_rmu_reset_periperal(RMU_PERH_USB);
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
	ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, ENABLE);
	usb_pin_init();
	ald_usb_high_speed_enable(false);
	ald_usb_otg_session_request(true);
	usbd_hid_boot_init(0, &hid_boot_device);

	while (1) {
		if (boot_complete == true)
		{
			app_size = *(volatile uint32_t *)(APP_FLASH_START_ADDRESS + actual_size - 4);
			app_crc  = *(volatile uint32_t *)(APP_FLASH_START_ADDRESS + actual_size - 8);
			
			calc_crc = check_app_crc((uint8_t *)APP_FLASH_START_ADDRESS, app_size);
			if (calc_crc == app_crc)
				application_check_and_run();
		}	
	}
}

/**
  * @}
  */
/**
  * @}
  */
