/**
  **************************************************************************************
  * @file    app_update.c
  * @brief   app_update Program Body.
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 
/* Includes-------------------------------------------------------------------*/

#include "app_update.h"
#include "string.h"
#include "ald_syscfg.h"
#include "ald_rmu.h"
#include "ald_cmu.h"
#include "ald_pmu.h"

uint32_t boot_complete = 0;

/** @addtogroup Private_Function
  * @{
  */  
  
/**
  * @brief  Testing the APP legitimacy and jump to the APP
  * @param  None
  * @retval None
  */
uint8_t application_check_and_run(void)
{
	uint32_t addr = 0;

	ald_delay_ms(100);
	ald_rmu_reset_periperal(RMU_PERH_USB);

	ald_pmu_perh_power_config(PMU_POWER_USB, DISABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, DISABLE);
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, DISABLE);
	ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, DISABLE);

	ald_rmu_reset_periperal(RMU_PERH_UART1);

	SYSCFG_UNLOCK();
	SYSCFG_BOOTFLASH_MAPPING_DISABLE();
	SYSCFG_LOCK();
	
	addr = *(uint32_t*)(APP_FLASH_START_ADDRESS + 4) ;

	__set_MSP(*(uint32_t *)APP_FLASH_START_ADDRESS);
	ald_vtor_config(APP_FLASH_START_ADDRESS, ENABLE);
	((void(*)(void))(addr))();


	return TRUE;
}

/**
  * @brief  erase app flash
  * @param  None
  * @retval TRUE/FALSE
  */
uint8_t app_flash_erase(void)
{
	uint32_t i = APP_FLASH_START_ADDRESS;

	for (i = APP_FLASH_START_ADDRESS; i < APP_FLASH_SIZE + APP_FLASH_START_ADDRESS; i += 1024) {
		if (ald_iap_erase_page(i) == ERROR)
			return FALSE;
	}

	return TRUE;
}
/**
  * @brief  write app flash
  * @param  addr: flash address
  * @param  data: data buff to write
  * @param  data_length: byte length to write
  * @retval TRUE/FALSE
  */
uint8_t app_flash_write(uint32_t addr, const uint8_t *data, uint32_t data_length)
{
	uint8_t buf[64];
	__disable_irq();
	memcpy((void*)buf, (const void*)data , data_length);

	if ((addr >= (APP_FLASH_START_ADDRESS + APP_FLASH_SIZE)) || (addr < APP_FLASH_START_ADDRESS))
		return FALSE;

	if (data_length & 0x07) {
		buf[data_length]     = 0xFF;
		buf[data_length + 1] = 0xFF;
		buf[data_length + 2] = 0xFF;
		buf[data_length + 3] = 0xFF;
		data_length += 4;
	}

	if (ald_iap_program_words(addr, buf, data_length, FALSE) != 0) {
		__enable_irq();
		return FALSE;
	}
	__enable_irq();
	return TRUE;  
}

/**
  * @brief  get app flash info
  * @param  start_addr: flash start address
  * @param  size: flash size
  * @retval None
  */
void app_flash_info(uint32_t *start_addr, uint32_t *size)
{
	if (start_addr)
		*start_addr = APP_FLASH_START_ADDRESS;

	if (size)
		*size =  APP_FLASH_SIZE;

	return;
}

