/**
  *********************************************************************************
  *
  * @file    usbd_hid_game.h
  * @brief   Header file for the HID game pad driver[Device].
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __USBD_HID_CUSTOM_H__
#define __USBD_HID_CUSTOM_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usbd_hid.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @addtogroup Device_HID_Boot
  * @{
  */
/**
  * @defgroup Device_HID_Boot_Public_Macros Public Macros
  * @{
  */
#define CUSTOMHID_REPORT_SIZE		64
#define CUSTOM_SUCCESS			0
#define CUSTOM_ERR_TX_ERROR		2
#define CUSTOM_ERR_RX_ERROR		3
#define CUSTOM_ERR_NOT_CONFIGURED	4
/**
  * @}
  */
/**
  * @defgroup Device_HID_Boot_Public_Types Public Types
  * @{
  */
/**
  * @brief HID boot pad status
  */
typedef enum {
	HID_BOOT_STATE_UNCONFIG  = 0,	/**< Unconfigured */
	HID_BOOT_STATE_IDLE      = 1,	/**< Idle */
	HID_BOOT_STATE_WAIT_DATA = 2,	/**< Waiting data */
	HID_BOOT_STATE_SEND      = 3,	/**< Send */
} boot_state_t;

/**
  * @brief HID Boot instance information
  */
typedef struct {
	uint8_t config;					/**< Configuration number */
	uint8_t protocol;				/**< Protocol */
	uint8_t txReport[64];				/**< IN report */
	uint8_t rxReport[64];				/**< OUT report */
	boot_state_t state;				/**< Status */
	hid_report_idle_t idle;				/**< Idle */
	usbd_hid_dev_t dev;				/**< HID device */
} hid_boot_inst_t;

/**
  * @brief Define operating parameters for the HID Boot device
  */
typedef struct {
	const uint16_t vid;			/**< Vendor ID */
	const uint16_t pid;			/**< Product ID */
	const uint16_t max_power;		/**< Maximum power consumption */
	const uint8_t attr_pwr;			/**< Attribute of the power */
	const usb_cbk cbk;			/**< Callback function */
	void *arg;				/**< Parameter */
	const uint8_t * const *desc_str;	/**< String descriptor */
	const uint32_t num_str;			/**< Number of the string descriptors */
	hid_boot_inst_t inst;			/**< Device instance */
} usbd_hid_boot_dev_t;

/**
  * @}
  */

/** @addtogroup Device_Game_Public_Functions
  * @{
  */
extern void *usbd_hid_boot_init(uint32_t idx, usbd_hid_boot_dev_t *dev);
extern void *usbd_hid_boot_init_comp(uint32_t idx, usbd_hid_boot_dev_t *dev, comp_entry_t *entry);
extern uint32_t usbd_hid_boot_report_send(usbd_hid_boot_dev_t *dev, void *report, uint32_t size);
extern uint32_t usbd_hid_boot_report_recv(usbd_hid_boot_dev_t *dev, void *report, uint32_t size);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
