/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"
#include "usbh_msc.h"
#include "boot_fatfs.h"

/** @addtogroup Projects_Examples_USB
  * @{
  */

/** @addtogroup Examples
  * @{
  */
#define APP_IN_FLASH
#define STATE_DEVICE_NO		0x01
#define STATE_DEVICE_ENUM	0x02
#define STATE_DEVICE_READY	0x04
#define STATE_DEVICE_UNKNOWN	0x08
#define STATE_POWER_FAULT	0x10

#define USBH_POOL_LEN	128

bool start_flag = false;
bool load_flag = false;
uint32_t FileSz = 0;
uint32_t StFAT = 0;
uint8_t *pBin;
uint32_t DataSec = 0;
uint32_t RootSec = 0;
bin_hdr_t bin_hdr = {0};
uint32_t rsv_sector = 0;
uint32_t App_addr = 0;
uint32_t cnt_tick = 0;
uint32_t tmp = 0;
uint16_t i = 0;

uint8_t bin_buf[1024] = {0};
uint32_t last_cnt = 0;

uint32_t rx_nr;
env_t env;
uint8_t tx_buf[512];
uint8_t rx_buf[512];
uint8_t usbh_pool[USBH_POOL_LEN];
uint32_t _msc_flag = STATE_DEVICE_NO;
usbh_msc_inst_t *inst;

void usb_hcd_event(void *pdata);
DECLARE_EVENT_DRIVER(usb_event_driver, 0, 0, usb_hcd_event);
static usbh_class_driver_t const * const __host_driver[] = {
	&__usbh_msc_driver,
	&usb_event_driver
};
static const uint32_t __nr_host_driver = sizeof(__host_driver) / sizeof(usbh_class_driver_t *);

/**
  * @brief  Initializate pin of USB.
  * @retval None
  */
void usb_pin_init(void)
{
	gpio_init_t x;

	/* Initialize vbus pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_5;
	ald_gpio_init(GPIOB, GPIO_PIN_15, &x);

	return;
}

/**
  * @brief  MSC callback.
  * @param  inst: MSC instance.
  * @param  event: Type of the event.
  * @param  data: message of the event.
  * @retval None
  */
void usr_msc_cbk(usbh_msc_inst_t *inst, uint32_t event, void *data)
{
	switch (event) {
	case MSC_EVENT_OPEN:
		printf_e("\rSTATE_DEVICE_ENUM\n\r");
		_msc_flag = STATE_DEVICE_ENUM;
		break;
	case MSC_EVENT_CLOSE:
		printf_e("\rMSC close\n\r");
		_msc_flag = STATE_DEVICE_NO;
		break;
	default:
		break;
	}
}

/**
  * @brief  Handle HCD event.
  * @param  data: Parameter of the event.
  * @retval None
  */
void usb_hcd_event(void *data)
{
	event_info_t *info = (event_info_t *)data;

	switch (info->event) {
	case USB_EVENT_CONNECTED:
		/* Device is connect */
		printf_e("\rDevice Connected\n\r");
		break;
	case USB_EVENT_DISCONNECTED:
		/* Device is disconnect */
		printf_e("\rDevice Disconnected\n\r");
		env.flag = 0;
		_msc_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_POWER_FAULT:
		printf_e("\rSTATE_POWER_FAULT\n\r");
		_msc_flag = STATE_POWER_FAULT;
		break;
	default:
		break;
	}
}

/**
  * @brief  Check app CRC
  * @param  page: total page of flash data.
  * @retval 0-success, other value indicates failed.
  */
uint32_t check_app_crc(uint8_t *src, uint32_t len)
{
    crc_handle_t h_crc;
   
    /* Initialize CRC */
    h_crc.perh          = CRC;
    h_crc.init.mode     = CRC_MODE_32;
    h_crc.init.seed     = 0xFFFFFFFF;
    h_crc.init.data_rev = ENABLE;
    h_crc.init.data_inv = DISABLE;
    h_crc.init.chs_rev  = ENABLE;
    h_crc.init.chs_inv  = ENABLE;
    ald_crc_init(&h_crc);
    
    return ald_crc_calculate(&h_crc, (uint8_t *)src, len);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	volatile uint32_t app_crc = 0;
	volatile uint32_t calc_crc = 0;
	volatile uint32_t app_size = 0;
	
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

	memset(&env, 0x0, sizeof(env_t));
	memset(tx_buf, 0x11, 64);
	memset(tx_buf + 64, 0x22, 64);
	memset(tx_buf + 64 * 2, 0x33, 64);
	memset(tx_buf + 64 * 3, 0x44, 64);
	memset(tx_buf + 64 * 4, 0x55, 64);
	memset(tx_buf + 64 * 5, 0x66, 64);
	memset(tx_buf + 64 * 6, 0x77, 64);
	memset(tx_buf + 64 * 7, 0x88, 64);
	memset(rx_buf, 0x0, 512);

	uart_stdio_init();
	printf_e("\rSystem start...\n\r");

	ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_CRC, ENABLE);
	ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
	ald_rmu_reset_periperal(RMU_PERH_USB);
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
	usb_pin_init();

	usb_hcd_register_driver(0, __host_driver, __nr_host_driver);
	inst = usbh_msc_driver_open(0, usr_msc_cbk);
	usb_hcd_init(0, usbh_pool, USBH_POOL_LEN);
	cnt_tick = ald_get_tick();

	while (1) {
		usb_hcd_main();
		/* receive update program timeout handle*/
		if (ald_get_tick() - cnt_tick > 5000) {
			if (get_run_flag(&FileSz) == RUN_IN_FLASH) {
				ald_rmu_reset_periperal(RMU_PERH_USB);
				printf_e("\rExecute intrinsic program in flash\n\r");
				run_in_flash();
			}
			else if (get_run_flag(&FileSz) == RUN_IN_RAM) {
				ald_rmu_reset_periperal(RMU_PERH_USB);
				app_ram_write((uint8_t *)APP_FLASH_S, FileSz);
				printf_e("\rExecute intrinsic program in ram\n\r");
				run_in_ram();
			}
			else {
				printf_e("\rWait update APP program\n\r");
			}
			cnt_tick = ald_get_tick();
		}

		switch (_msc_flag) {
		case STATE_DEVICE_ENUM:
			if ((usbh_msc_driver_ready(inst)) != 0)
				break;

			_msc_flag = STATE_DEVICE_READY;
			break;

		case  STATE_DEVICE_READY:
			if (env.flag)
				break;

			env.flag = 1;
			printf_e("\rDevice ready!\n\r");
			printf_e("\rRead DBR section!\n\r");
			memset(tx_buf, 0, sizeof(tx_buf));
			cnt_tick = ald_get_tick();

			if (!usbh_msc_block_read(inst, 1, rx_buf, 1)) {
				//FAT32
				if ((*(uint32_t*)rx_buf == 0x41615252) && (*(uint32_t*)(&rx_buf[484]) == 0x61417272)) {
					usbh_msc_block_read(inst, 0, rx_buf, 1);
				} else {		
					usbh_msc_block_read(inst, 0, rx_buf, 1);
					rsv_sector = rx_buf[454];
					usbh_msc_block_read(inst, rsv_sector, rx_buf, 1);
					FATFS_TYPE = TYPE_FAT16;
				}

				/* check fatfs type */
				get_fatfs_type(rx_buf);
				RootSec = get_rootclust_sec(rx_buf) + rsv_sector;
				/* check update program is existing */
				usbh_msc_block_read(inst, RootSec, rx_buf, 1);
				StFAT = get_dir_param(rx_buf, binName, &FileSz);
				/* get update program section */
				if (FATFS_TYPE == TYPE_FAT32) {
					usbh_msc_block_read(inst, 0, rx_buf, 1);
				} else {
					usbh_msc_block_read(inst, rsv_sector, rx_buf, 1);
				}
				DataSec = get_curfat_sec(rx_buf, StFAT) + rsv_sector;
				/* get update program head info struct */
				usbh_msc_block_read(inst, DataSec, rx_buf, 1);
				pBin = get_bin_hdr(rx_buf, &bin_hdr, DataSec);
				last_cnt = FileSz % DISK_BLOCK_SIZE;

				tmp = FileSz / DISK_BLOCK_SIZE;

				if (FileSz <= 512) {
					printf_e("\rCan't find file\n\r");
					break;
				}

				/* load update program */
				for (i = 0; i < tmp; i++) {
					memset(bin_buf, 0xFF, sizeof(bin_buf));
					usbh_msc_block_read(inst, DataSec, bin_buf, 2);

					App_addr = APP_FLASH_S + i * 1024;
					ald_iap_program_words(App_addr, bin_buf, 1024, ENABLE);
					DataSec += 2;
				}

				if (FileSz != (tmp * DISK_BLOCK_SIZE)) {
					memset(bin_buf, 0xFF, sizeof(bin_buf));
					usbh_msc_block_read(inst, DataSec, bin_buf, 2);

					App_addr += 1024;
					ald_iap_program_words(App_addr, bin_buf, 1024, ENABLE);
				}
				// ADD CRC
				app_size = *(volatile uint32_t *)(APP_FLASH_S + FileSz - 4);
				app_crc  = *(volatile uint32_t *)(APP_FLASH_S + FileSz - 8);
				#ifdef APP_IN_FLASH
				calc_crc = check_app_crc((uint8_t *)APP_FLASH_S, app_size);
				#else
				app_ram_write((uint8_t *)APP_FLASH_S, FileSz);
				calc_crc = check_app_crc((uint8_t *)APP_RAM_S, app_size);
				#endif
				if (calc_crc != app_crc) 
					break;
			}
#ifdef APP_IN_FLASH
			App_addr = APP_FLASH_S + FileSz;
			if ((App_addr >= APP_FLASH_S) && (App_addr <= APP_FLASH_E)) {
				set_run_flag(RUN_IN_FLASH, FileSz);
				ald_rmu_reset_periperal(RMU_PERH_USB);
				printf_e("\rRun in flash\n\r");
				run_in_flash();
			}
#else
			App_addr = APP_RAM_S + FileSz;
			if ((App_addr >= APP_RAM_S) && (App_addr <= APP_RAM_E)) {
				set_run_flag(RUN_IN_RAM, FileSz);
				ald_rmu_reset_periperal(RMU_PERH_USB);
				printf_e("\rRun in ram\n\r");
				run_in_ram();
			}
#endif

			printf_e("\rApp address Error!\n\r");

			if (get_run_flag(&FileSz) == RUN_IN_FLASH) {
				ald_rmu_reset_periperal(RMU_PERH_USB);
				printf_e("\rExecute intrinsic program in flash\n\r");
				run_in_flash();
			}
			else if (get_run_flag(&FileSz) == RUN_IN_RAM) {
				ald_rmu_reset_periperal(RMU_PERH_USB);
				app_ram_write((uint8_t *)APP_FLASH_S, FileSz);
				printf_e("\rExecute intrinsic program in ram\n\r");
				run_in_ram();
			}
			else {
				printf_e("\rWait update APP program\n\r");
			}

			break;

		case STATE_DEVICE_NO:
		case STATE_DEVICE_UNKNOWN:
		case STATE_POWER_FAULT:
			break;

		default:
			break;
		}
}
}

/**
  * @}
  */
/**
  * @}
  */
