/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"
#include "ald_conf.h"
#include "usbd_msc.h"
#include "boot_fatfs.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
#define APP_IN_FLASH
#define STATE_DEVICE_NO		0x01
#define STATE_DEVICE_CONN	0x02
#define STATE_DEVICE_READY	0x04
#define STATE_DEVICE_READ	0x08
#define STATE_DEVICE_WRITE	0x10
#define STATE_DEVICE_IDLE	0x20
#define STATE_POWER_FAULT	0x40

bool start_flag = false;
bool load_flag = false;
uint32_t FileSz = 0;
uint32_t StFAT = 0;
uint8_t *pBin = NULL;
bin_hdr_t bin_hdr = {0};
uint32_t App_addr = 0;
volatile uint32_t cnt_tick = 0;
volatile uint32_t __expire = 0;

uint32_t rx_nr;
env_t env;
uint8_t tx_buf[1024];
uint8_t rx_buf[1024];
uint32_t idx_media = 1;
uint32_t _msc_flag = STATE_DEVICE_NO;

uint32_t msc_event_handle(void *data, uint32_t event, uint32_t value, void *p_data);

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void usb_pin_init(void)
{
	gpio_init_t x;

	/* Initialize HSCO pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_20;
	x.nodrv = GPIO_OUT_DRIVE_20;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_6;
	ald_gpio_init(GPIOA, GPIO_PIN_15, &x);

	return;
}

const uint8_t lang_desc[] =
{
	4,
	USB_DTYPE_STRING,
	USBShort(USB_LANG_EN_US)
};

const uint8_t manufact_str[] = {
	(17 + 1) * 2,   
	USB_DTYPE_STRING,
	'E', 0, 'a', 0, 's', 0, 't', 0, 's', 0, 'o', 0, 'f', 0, 't', 0, ' ', 0,
	'S', 0, 'h', 0, 'a', 0, 'n', 0, 'g', 0, 'h', 0, 'a', 0, 'i', 0
};

const uint8_t product_str[] = {
	(19 + 1) * 2,
	USB_DTYPE_STRING,
	'M', 0, 'a', 0, 's', 0, 's', 0, ' ', 0, 'S', 0, 't', 0, 'o', 0,
	'r', 0, 'a', 0, 'g', 0, 'e', 0, ' ', 0, 'D', 0, 'e', 0, 'v', 0,
	'i', 0, 'c', 0, 'e', 0
};

const uint8_t serial_num_str[] =
{
	(8 + 1) * 2,
	USB_DTYPE_STRING,
	'1', 0, '2', 0, '3', 0, '4', 0, '5', 0, '6', 0, '7', 0, '8', 0
};

const uint8_t data_interface_str[] =
{
	(19 + 1) * 2,
	USB_DTYPE_STRING,
	'B', 0, 'u', 0, 'l', 0, 'k', 0, ' ', 0, 'D', 0, 'a', 0, 't', 0,
	'a', 0, ' ', 0,	'I', 0, 'n', 0, 't', 0, 'e', 0, 'r', 0, 'f', 0,
	'a', 0, 'c', 0, 'e', 0
};

const uint8_t config_str[] =
{
	(23 + 1) * 2,
	USB_DTYPE_STRING,
	'B', 0, 'u', 0, 'l', 0, 'k', 0, ' ', 0, 'D', 0, 'a', 0, 't', 0,
	'a', 0, ' ', 0,	'C', 0, 'o', 0, 'n', 0, 'f', 0, 'i', 0, 'g', 0,
	'u', 0, 'r', 0, 'a', 0, 't', 0, 'i', 0, 'o', 0, 'n', 0
};

const uint8_t *const string_desc[] =
{
	lang_desc,
	manufact_str,
	product_str,
	serial_num_str,
	data_interface_str,
	config_str
};

#define NUM_STRING_DESCRIPTORS (sizeof(string_desc) / sizeof(uint8_t *))

/* Interrupt environment */
void usr_bin_transmit_cplt(void)
{
	start_flag = true;
	return;
}

void ald_systick_irq_cbk(void)
{
	if (--__expire == 0)
		usr_bin_transmit_cplt();
}

void usr_timer_start(uint32_t cnt)
{
	__expire = cnt;
}

void *msc_open(uint32_t device)
{
	return (void *)&idx_media;
}

void msc_close(void *device)
{
	return;
}

uint32_t msc_read(void *device, uint8_t *buf, uint32_t sector, uint32_t num_block)
{
	fs_flash_read(buf, sector, num_block);
	return num_block * DISK_BLOCK_SIZE;
}

uint32_t msc_write(void *device, uint8_t *buf, uint32_t sector, uint32_t num_block)
{
	fs_flash_write(buf, sector, num_block);
	usr_timer_start(3000);
	return num_block * DISK_BLOCK_SIZE;
}

uint32_t msc_get_block_num(void *device)
{
	return DISK_BLOCK_NR;
}

uint32_t msc_get_block_size(void *device)
{
	return DISK_BLOCK_SIZE;
}

usbd_msc_dev_t msc_device = {
	USB_VID_EASTSOFT_30CC,
	USB_PID_MSC,
	"EASTSOFT",
	"Mass Storage    ",
	"1.00",
	500,
	USB_CONF_ATTR_SELF_PWR,
	string_desc,
	NUM_STRING_DESCRIPTORS,
	{
		msc_open,
		msc_close,
		msc_read,
		msc_write,
		msc_get_block_num,
		msc_get_block_size,
	},
	msc_event_handle,
};

uint32_t msc_event_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
	switch (event) {
	case USB_EVENT_DISCONNECTED:
		_msc_flag = STATE_DEVICE_NO;
		break;
	case USB_EVENT_CONNECTED:
		_msc_flag = STATE_DEVICE_CONN;
		load_flag = true;

		break;
	case USBD_MSC_EVENT_WRITING:
		_msc_flag = STATE_DEVICE_WRITE;
		break;
	case USBD_MSC_EVENT_READING:
		_msc_flag = STATE_DEVICE_READ;
		break;
	case USBD_MSC_EVENT_IDLE:
		_msc_flag = STATE_DEVICE_IDLE;
		break;
	default:
		break;
	}

	return 0;
}

/**
  * @brief  Check app CRC
  * @param  page: total page of flash data.
  * @retval 0-success, other value indicates failed.
  */
uint32_t check_app_crc(uint8_t *src, uint32_t len)
{
    crc_handle_t h_crc;
   
    /* Initialize CRC */
    h_crc.perh          = CRC;
    h_crc.init.mode     = CRC_MODE_32;
    h_crc.init.seed     = 0xFFFFFFFF;
    h_crc.init.data_rev = ENABLE;
    h_crc.init.data_inv = DISABLE;
    h_crc.init.chs_rev  = ENABLE;
    h_crc.init.chs_inv  = ENABLE;
    ald_crc_init(&h_crc);
    
    return ald_crc_calculate(&h_crc, (uint8_t *)src, len);
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	volatile uint32_t app_crc = 0;
	volatile uint32_t calc_crc = 0;
	volatile uint32_t app_size = 0;
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

	memset(&env, 0x0, sizeof(env_t));
	memset(tx_buf, 0x0, 512);
	memset(rx_buf, 0x0, 512);

	uart_stdio_init();
	printf_e("\rSystem start...\n\r");

	init_fat12_fs();

	ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_CRC, ENABLE);
	ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
	ald_rmu_reset_periperal(RMU_PERH_USB);
	ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
	ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, ENABLE);
	usb_pin_init();
	ald_usb_high_speed_enable(false);
	ald_usb_otg_session_request(true);
	usbd_msc_init(0, &msc_device);
	cnt_tick = ald_get_tick();

	while (1) {
		if ((load_flag == true) && (start_flag == false))
			continue;

		if (start_flag) {
			start_flag = false;

			get_fatfs_type((uint8_t *)FAT_FLASH_S);
			StFAT = get_dir_param((uint8_t *)FAT_FLASH_S, binName, &FileSz);
			pBin = get_bin_hdr((uint8_t *)FAT_FLASH_S, &bin_hdr, StFAT);

			if ((StFAT == 0) || (pBin == NULL))
				continue;

			app_size = *(volatile uint32_t *)(pBin + FileSz - 4);
			app_crc  = *(volatile uint32_t *)(pBin + FileSz - 8);
			
			#ifdef APP_IN_FLASH
			app_flash_write(pBin, FileSz);
			calc_crc = check_app_crc((uint8_t *)APP_FLASH_S, app_size);
			#else
			app_ram_write((uint8_t *)pBin, FileSz);
			calc_crc = check_app_crc((uint8_t *)APP_RAM_S, app_size);
			#endif
			if (calc_crc != app_crc) 
				continue;
#ifdef APP_IN_FLASH
			App_addr = APP_FLASH_S + FileSz;
			if ((App_addr >= APP_FLASH_S) && (App_addr <= APP_FLASH_E)) {
				set_run_flag(RUN_IN_FLASH, FileSz);
				ald_rmu_reset_periperal(RMU_PERH_USB);
				printf_e("\rRun in flash\n\r");
				run_in_flash();
			}
#else
			App_addr = APP_RAM_S + FileSz;
			if ((App_addr >= APP_RAM_S) && (App_addr <= APP_RAM_E)) {
				
				set_run_flag(RUN_IN_RAM, FileSz);
				ald_rmu_reset_periperal(RMU_PERH_USB);
				printf_e("\rRun in ram\n\r");
				run_in_ram();
			}
#endif
			printf_e("\rApp address Error!\n\r");
		}
		else {
			if (ald_get_tick() - cnt_tick > 5000) {
				cnt_tick = ald_get_tick();
				if (get_run_flag(&FileSz) == RUN_IN_FLASH) {
					ald_rmu_reset_periperal(RMU_PERH_USB);
					printf_e("\rRun in flash\n\r");
					run_in_flash();
				}
				else if (get_run_flag(&FileSz) == RUN_IN_RAM) {
					ald_rmu_reset_periperal(RMU_PERH_USB);
					app_ram_write((uint8_t *)APP_FLASH_S, FileSz);
					printf_e("\rRun in ram\n\r");
					run_in_ram();
				}
				else {
					printf_e("\rWait update APP program\n\r");
				}
			}
		}
	}
}

/**
  * @}
  */
/**
  * @}
  */
