/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for demo
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

md_uart_init_t uart_init;
md_uart_handle_t h_uart;
uint8_t rtx_buf[100] = {0};
uint8_t rframe = 0;
uint8_t sflag = 0;


/**
  * @brief  Init UART pin
  * @retval None
  */
void md_uart_pin_init(void)
{
    md_gpio_init_t x;
    /* Initialize tx pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART0_TX_PORT, UART0_TX_PIN, &x);

    /* Initialize rx pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART0_RX_PORT, UART0_RX_PIN, &x);
}


/**
  * @brief  This function handles UART interrupt request.
  * @param  hperh: Pointer to a md_uart_handle_t structure.
  * @retval None
  */
void md_uart_irq_handler(UART_TypeDef *UARTx)
{
    uint8_t tmp = 0;

    /* Receive */
    if (md_uart_is_trigger_fifo_threshold_rfth(UARTx))
    {
        while (md_uart_get_rec_rxfl(UARTx))
        {
            if (h_uart.rx_count < h_uart.rx_size)
            {
                *h_uart.rx_buf++ = md_uart_recv_data8(UARTx);
                h_uart.rx_count++;
            }
            else
            {
                tmp = md_uart_recv_data8(UARTx);
                UNUSED(tmp);
            }
        }
        md_uart_clear_it_rfth(UART0);
        if (h_uart.rx_count >= h_uart.rx_size)
        {
            md_uart_disable_it_rfth(UARTx);
            // User code
        }
    }

    /* Transmit */
    if (md_uart_mask_it_tfth(UARTx))
    {
        while (md_uart_is_active_flag_tffull(UARTx));

        UARTx->TXBUF = (uint8_t)(*h_uart.tx_buf++);
        h_uart.tx_count++;

        if (h_uart.tx_count >= h_uart.tx_size)
        {
            md_uart_disable_it_tfth(UARTx);
            md_uart_enable_it_tbc(UARTx);
        }
    }

    /* End Transmit */
    if ((md_uart_mask_it_tbc(UARTx)) != RESET)
    {
        md_uart_clear_it_tbc(UARTx);
        md_uart_disable_it_tbc(UARTx);
        
        sflag = 0x1;
    }

    /* Receive frame timeout */
    if ((md_uart_mask_it_rxto(UARTx)) != RESET)
    {
        md_uart_clear_it_rfth(UART0);
        md_uart_clear_flag_rxto(UARTx);
        md_uart_disable_it_rfth(UART0);   
        md_uart_disable_it_rxto(UART0);
        
        rframe = 0x1;
    }
}

/**
  * @brief  Receive timeout settings.
  * @param  buf: receive target start address.
  * @param  Maxlen: receive maxim length. 
  * @param  tout: Baud rate's bit time duration.
  * @retval None
  */
void rec_rto(uint8_t *buf, uint8_t MAXlen, uint32_t tout)
{
    h_uart.rx_count = 0;
    h_uart.rx_buf = rtx_buf;
    h_uart.rx_size = MAXlen;
    
    md_uart_enable_it_rfrst(UART0);
    md_uart_clear_it_rfth(UART0);
    md_uart_enable_rtoen(UART0);
    /* config RTO time(bit duration) */ 
    md_uart_set_rec_timeout(UART0, tout);    
    md_uart_enable_it_rfth(UART0);   
    md_uart_enable_it_rxto(UART0);
}

/**
  * @brief  Send config.
  * @param  buf: send start address.
  * @param  len: send length. 
  * @retval None
  */
void send(uint8_t *buf, uint32_t len)
{
    h_uart.tx_count = 0;
    h_uart.tx_buf = buf;
    h_uart.tx_size = len;

    md_uart_clear_it_tfth(UART0);
    md_uart_enable_it_tfth(UART0);
    md_uart_enable_it_tfrst(UART0);
    md_uart_enable_tx(UART0);
}
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(2);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Initialize uart communication gpio */
    md_uart_pin_init();
    md_uart_init_struct(&uart_init);
    md_uart_init(UART0, &uart_init);
    md_mcu_irq_config(UART0_IRQn, 1, 2, ENABLE);
    memset(rtx_buf, 0x55, sizeof(rtx_buf));
    /* send 10 bytes */
    send(rtx_buf, 10);
    /* receive interrupt */
    rec_rto(rtx_buf, 100, 0xFF);

    while (1)
    {
        if (0x1 == rframe)
        {
            rframe = 0;
            send(rtx_buf, h_uart.rx_count);
            while(!sflag);
            sflag = 0;
            memset(rtx_buf, 0x00, sizeof(rtx_buf));
            rec_rto(rtx_buf, 100, 0xFF);
        }
    }
}

