/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  Initialize LED2.
  * @param  None
  * @retval None
  */
void gpio_init(void)
{
    md_gpio_init_t gpio_init;

    md_gpio_set_pin_function(LED_GPIO_PORT, LED_GPIO_PIN, MD_GPIO_FUNC_1);
    md_gpio_set_pin_push_pull(LED_GPIO_PORT, LED_GPIO_PIN);
    md_gpio_set_pin_mode_output(LED_GPIO_PORT, LED_GPIO_PIN);

    md_gpio_init_struct(&gpio_init);
    gpio_init.mode = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.pupd = MD_GPIO_FLOATING;
    gpio_init.func = MD_GPIO_FUNC_2;

    md_gpio_init(CH1_PORT, CH1_PIN, &gpio_init);
    md_gpio_init(CH2_PORT, CH2_PIN, &gpio_init);
    md_gpio_init(CH3_PORT, CH3_PIN, &gpio_init);
    md_gpio_init(CH1N_PORT, CH1N_PIN, &gpio_init);
    md_gpio_init(CH2N_PORT, CH2N_PIN, &gpio_init);
    md_gpio_init(CH3N_PORT, CH3N_PIN, &gpio_init);

}

/**
  * @brief:  Initialize AD16C4T to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t ad16c4t_init;
    md_timer_oc_init_t oc_init;

    md_timer_base_struct_init(&ad16c4t_init);           /* initialize the ad16c4t_init  */
    ad16c4t_init.prescaler = 48 - 1;                    /* clk_count: 48 */
    ad16c4t_init.clk_div = MD_TIMER_CLOCK_DIV1;         /* working clock of dead time and filter */
    ad16c4t_init.mode = MD_TIMER_CNT_MODE_CENTER1;      /* count up */
    ad16c4t_init.period = 5000 - 1;                     /* period is 5000 count clock */
    ad16c4t_init.re_cnt = 1 - 1;                        /* 0 repeat count */
    md_timer_base_set_config(AD16C4T0, &ad16c4t_init);

    md_timer_oc_struct_init(&oc_init);
    oc_init.ocstate = MD_TIMER_OCSTATE_ENABLE;          /* enable CHxO compare output */
    oc_init.ocnstate = MD_TIMER_OCSTATE_ENABLE;         /* enable CHxON compare output */
    oc_init.oc_mode = MD_TIMER_OC_MODE_PWM2;            /* compare output mode is PWM2 */
    oc_init.oc_polarity = MD_TIMER_OC_POLARITY_HIGH;    /* CHxO compare output polarity is high */
    oc_init.ocn_polarity = MD_TIMER_OCN_POLARITY_HIGH;  /* CHxON compare output polarity is high */
    oc_init.oc_idle = MD_TIMER_OC_IDLE_RESET;           /* CHxO idle is low */
    oc_init.ocn_idle = MD_TIMER_OCN_IDLE_RESET;         /* CHxON idle is low */
    oc_init.pulse = 1500 - 1;                           /* CH1 compare value is 1500 - 1 */
    md_timer_oc_init(AD16C4T0, MD_TIMER_CHANNEL_1, &oc_init);
    oc_init.pulse = 2500 - 1;                           /* CH2 compare value is 2500 - 1 */
    md_timer_oc_init(AD16C4T0, MD_TIMER_CHANNEL_2, &oc_init);
    oc_init.pulse = 3500 - 1;                           /* CH3 compare value is 3500 - 1 */
    md_timer_oc_init(AD16C4T0, MD_TIMER_CHANNEL_3, &oc_init);

    md_timer_enable_cc1it_interrupt(AD16C4T0);          /* enable CC1 interrupt request */
    md_timer_enable_cc2it_interrupt(AD16C4T0);          /* enable CC2 interrupt request */
    md_timer_enable_cc3it_interrupt(AD16C4T0);          /* enable CC3 interrupt request */
    md_timer_enable_uit_interrupt(AD16C4T0);            /* update interrupt request */

    md_timer_set_main_output_enable(AD16C4T0, ENABLE);       /* enable output */
    md_timer_enable_counter_cnten(AD16C4T0);                 /* enable AD16C4T0 */
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(5);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Initialize pin */
    gpio_init();
    /* Initialize timer */
    init_timer();

    md_mcu_irq_config(AD16C4T0_UP_IRQn, 1, 1, ENABLE);
    md_mcu_irq_config(AD16C4T0_CC_IRQn, 1, 1, ENABLE);

    while (1)
    {
        md_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */
