/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

uint16_t g_duty_buffer[BUFFER_SIZE];
uint16_t g_cycle_buffer[BUFFER_SIZE];


/** @addtogroup Examples
  * @{
  */

/**
  * @brief  Initialize gpio.
  * @param  None
  * @retval None
  */
void gpio_init(void)
{
    md_gpio_init_t gpio_init;

    md_gpio_set_pin_function(LED_GPIO_PORT, LED_GPIO_PIN, MD_GPIO_FUNC_1);
    md_gpio_set_pin_push_pull(LED_GPIO_PORT, LED_GPIO_PIN);
    md_gpio_set_pin_mode_output(LED_GPIO_PORT, LED_GPIO_PIN);

    md_gpio_init_struct(&gpio_init);
    gpio_init.mode = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.pupd = MD_GPIO_FLOATING;
    gpio_init.func = MD_GPIO_FUNC_2;
    md_gpio_init(CH1_PORT, CH1_PIN, &gpio_init);
}

void dma_init(void)
{
    md_dma_config_t cycle_config;
    md_dma_config_t duty_config;

    /* DMA cycle config */
    memset(&cycle_config, 0x0, sizeof(md_dma_config_t));
    md_dma_config_struct(&cycle_config);
    cycle_config.mem_to_mem     = DISABLE;
    cycle_config.channel        = 0;
    cycle_config.dst_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    cycle_config.dst_inc        = MD_DMA_DATA_INC_ENABLE;
    cycle_config.src_inc        = MD_DMA_DATA_INC_DISABLE;
    cycle_config.src_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    cycle_config.src            = (void *)(&AD16C4T0->CCVAL1);
    cycle_config.dst            = (void *)&g_cycle_buffer;
    cycle_config.size           = BUFFER_SIZE;
    cycle_config.msel           = MD_DMA_MSEL_TIMER0;
    cycle_config.msigsel        = MD_DMA_MSIGSEL_TIMER_CH1;
    cycle_config.dir            = MD_DMA_DIR_TO_SRAM;
    cycle_config.circle_mode    = ENABLE;
    md_dma_init(DMA0, 0, &cycle_config);
    md_adc_converter_enable_adcen(ADC0);
    md_dma_enable_channel(DMA0, 0);
    md_adc_dma_enable(ADC0);
    /* Enable DMA0 channel0 TC interrupt */
    md_dma_enable_it_tc(DMA0, 0);

    /* DMA duty config */
    memset(&duty_config, 0x0, sizeof(md_dma_config_t));
    md_dma_config_struct(&duty_config);
    duty_config.mem_to_mem     = DISABLE;
    duty_config.channel        = 1;
    duty_config.dst_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    duty_config.dst_inc        = MD_DMA_DATA_INC_ENABLE;
    duty_config.src_inc        = MD_DMA_DATA_INC_DISABLE;
    duty_config.src_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    duty_config.src            = (void *)(&AD16C4T0->CCVAL2);
    duty_config.dst            = (void *)&g_duty_buffer;
    duty_config.size           = BUFFER_SIZE;
    duty_config.msel           = MD_DMA_MSEL_TIMER0;
    duty_config.msigsel        = MD_DMA_MSIGSEL_TIMER_CH2;
    duty_config.dir            = MD_DMA_DIR_TO_SRAM;
    duty_config.circle_mode    = ENABLE;
    md_dma_init(DMA0, 1, &duty_config);
    md_adc_converter_enable_adcen(ADC0);
    md_dma_enable_channel(DMA0, 1);
    md_adc_dma_enable(ADC0);
    /* Enable DMA0 channel0 TC interrupt */
    md_dma_enable_it_tc(DMA0, 1);
}

/**
  * @brief  Initialize TIMER.
  * @param  None
  * @retval None
  */
void timer_base_init(void)
{
    md_timer_base_init_t ad16c4t_init;
    md_timer_ic_init_t ic_init;

    md_timer_base_struct_init(&ad16c4t_init);   /* initialize the ad16c4t_init  */
    ad16c4t_init.prescaler = 48 - 1;            /* clk_count: 1MHz */
    ad16c4t_init.clk_div = MD_TIMER_CLOCK_DIV1; /* working clock of dead time and filter */
    ad16c4t_init.mode = MD_TIMER_CNT_MODE_UP;   /* count up */
    ad16c4t_init.period = 0xFFFF;               /* set max period */
    ad16c4t_init.re_cnt = 1 - 1;                /* 0 repeat count */
    md_timer_base_set_config(AD16C4T0, &ad16c4t_init);

    md_timer_ic_struct_init(&ic_init);                 /* initialize the ic_init  */
    ic_init.filter      = 0;                           /* 0 filter */
    ic_init.polarity    = MD_TIMER_IC_POLARITY_RISE;   /* capture rising edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;        /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_DIRECT;      /* capture this channel */
    md_timer_ic_init(AD16C4T0, MD_TIMER_CHANNEL_1, &ic_init);
    md_timer_ic_struct_init(&ic_init);
    ic_init.filter      = 0;
    ic_init.polarity    = MD_TIMER_IC_POLARITY_FALL;   /* capture falling edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;        /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_INDIRECT;    /* capture adjacent channel */
    md_timer_ic_init(AD16C4T0, MD_TIMER_CHANNEL_2, &ic_init);

    md_timer_set_slave_mode_smods(AD16C4T0, 4);        /* 0x4: Reset Mode */
    md_timer_set_slave_trigger_tssel(AD16C4T0, 5);     /* 0x5: Filtered Timer Input 1 */

    md_timer_set_cc_dma_select_ccdmasel(AD16C4T0, 0);  /* CCn DMA request sent when CCx event occurs */
    md_timer_enable_cc1dma(AD16C4T0);                  /* enable CC1 DMA request */
    md_timer_enable_cc2dma(AD16C4T0);                  /* enable CC2 DMA request */

    md_timer_enable_cc1it_interrupt(AD16C4T0);         /* enable CC1 interrupt request */
    md_timer_enable_cc2it_interrupt(AD16C4T0);         /* enable CC1 interrupt request */
    md_timer_enable_uit_interrupt(AD16C4T0);           /* enable update interrupt request */
    md_mcu_irq_config(AD16C4T0_UP_IRQn, 1, 1, ENABLE);
    md_mcu_irq_config(AD16C4T0_CC_IRQn, 1, 1, ENABLE);

    /* enable AD16C4T0 */
    md_timer_enable_counter_cnten(AD16C4T0);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(2);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Initialize pin */
    gpio_init();
    dma_init();
    /* Initialize timer */
    timer_base_init();

    while (1)
    {
        md_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */
