/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

md_spi_init_t h_spi;
md_dma_config_t g_dma_tx_config, g_dma_rx_config;
uint8_t g_recv_buf[10] = {0U};
uint8_t g_send_buf[10] = {0U};
uint8_t g_send_len = sizeof(g_send_buf);
uint8_t g_recv_len = sizeof(g_recv_buf);

uint8_t g_complete = 1;

/**
  * @brief  Initializate pin of spi module.
  * @retval None
  */
void spi_pin_init(void)
{
    md_gpio_init_t x;

    /* Initialize nss pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_1;
    md_gpio_init(GPIOA, MD_GPIO_PIN_4, &x);

    /* Initialize sck pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_6;
    md_gpio_init(GPIOA, MD_GPIO_PIN_5, &x);

    /* Initialize miso pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_6;
    md_gpio_init(GPIOA, MD_GPIO_PIN_6, &x);

    /* Initialize mosi pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_6;
    md_gpio_init(GPIOA, MD_GPIO_PIN_7, &x);

    return;
}

/**
  * @brief  dma_start_tx init.
  * @param  None
  * @retval None
  */
void dma_send_tx(void)
{
    md_dma_config_t dma_tx_config;

    /* Enable DMA0 channel 0 TC interrupt */
    md_dma_enable_it_tc(DMA0, MD_DMA_CH_0);

    memset(&g_dma_tx_config, 0x0, sizeof(md_dma_config_t));
    dma_tx_config.src            = (void *)g_send_buf;
    dma_tx_config.dst            = (void *)&SPI0->DATA;
    dma_tx_config.size           = g_send_len;
    dma_tx_config.src_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_tx_config.dst_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_tx_config.src_inc        = MD_DMA_DATA_INC_ENABLE;
    dma_tx_config.dst_inc        = MD_DMA_DATA_INC_DISABLE;
    dma_tx_config.R_power        = MD_DMA_R_POWER_1;    /* Must MD_DMA_R_POWER_1 */
    dma_tx_config.priority       = MD_DMA_LOW_PRIORITY;
    dma_tx_config.mem_to_mem     = DISABLE;
    dma_tx_config.circle_mode    = DISABLE;
    dma_tx_config.dir            = MD_DMA_DIR_TO_PERH;
    dma_tx_config.msel           = MD_DMA_MSEL_SPI0;
    dma_tx_config.msigsel        = MD_DMA_MSIGSEL_SPI_TXEMPTY;
    md_dma_init(DMA0, MD_DMA_CH_0, &dma_tx_config);
    md_dma_set_channel_single(DMA0, MD_DMA_CH_0);
}

/**
  * @brief  dma_start_rx init.
  * @param  None
  * @retval None
  */
void dma_recv_rx(void)
{
    /* Enable DMA0 channel 1 TC interrupt */
    md_dma_enable_it_tc(DMA0, MD_DMA_CH_1);

    memset(&g_dma_rx_config, 0x0, sizeof(md_dma_config_t));
    g_dma_rx_config.src            = (void *)&SPI0->DATA;
    g_dma_rx_config.dst            = (void *)g_recv_buf;
    g_dma_rx_config.size           = g_recv_len;
    g_dma_rx_config.src_data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_rx_config.dst_data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_rx_config.src_inc        = MD_DMA_DATA_INC_DISABLE;
    g_dma_rx_config.dst_inc        = MD_DMA_DATA_INC_ENABLE;
    g_dma_rx_config.R_power        = MD_DMA_R_POWER_1;           /* Must MD_DMA_R_POWER_1 */
    g_dma_rx_config.priority       = MD_DMA_LOW_PRIORITY;
    g_dma_rx_config.mem_to_mem     = DISABLE;
    g_dma_rx_config.circle_mode    = DISABLE;
    g_dma_rx_config.dir            = MD_DMA_DIR_TO_SRAM;
    g_dma_rx_config.msel           = MD_DMA_MSEL_SPI0;
    g_dma_rx_config.msigsel        = MD_DMA_MSIGSEL_SPI_RNR;
    md_dma_init(DMA0, MD_DMA_CH_1, &g_dma_rx_config);
    md_dma_set_channel_single(DMA0, MD_DMA_CH_1);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(2);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();
    md_mcu_irq_config(DMA0_IRQn, 1, 1, ENABLE);

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* SPI0 pin init */
    spi_pin_init();
    /* SPI0 peripheral init */
    memset(&h_spi, 0, sizeof(h_spi));
    md_spi_struct_init(&h_spi);
    h_spi.SPIx = SPI0;
    md_spi_init(&h_spi);
    md_gpio_write_pin(GPIOA, MD_GPIO_PIN_4, 1);

    while (1)
    {
        if (g_complete == 0x1)
        {
            g_complete = 0;
            md_gpio_write_pin(GPIOA, MD_GPIO_PIN_4, 0);
            /* receive */
            {
                dma_recv_rx();
                md_dma_enable_channel(DMA0, MD_DMA_CH_1);
                md_spi_enable_rxdma(SPI0);
            }
            /* send */
            {
                memcpy(g_send_buf, g_recv_buf, g_recv_len);
                dma_send_tx();
                md_dma_enable_channel(DMA0, MD_DMA_CH_0);
                md_spi_enable_txdma(SPI0);

                md_delay_1ms(1000);
            }
            md_gpio_write_pin(GPIOA, MD_GPIO_PIN_4, 1);
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */
