/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

md_rtc_time_t time;
md_rtc_date_t date;
char uart_buf[64];
md_uart_init_t h_uart;
/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
    uint8_t i, k;
    va_list args;

    va_start(args, fmt);
    vsnprintf(uart_buf, 64, fmt, args);
    va_end(args);

    k = strlen(uart_buf);

    for (i = 0; i < k; i++)
    {
        md_uart_set_send_data8(UART0, uart_buf[i]);

        while (!md_uart_is_trigger_fifo_threshold_tfth(UART0));
    }

    return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
    md_gpio_init_t x;

    /* Initialize tx pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOB, MD_GPIO_PIN_10, &x);

    /* Initialize rx pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOB, MD_GPIO_PIN_11, &x);

    /* Initialize uart */
    md_uart_init_struct(&h_uart);
    md_uart_init(UART0, &h_uart);

    return;
}
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    md_rtc_init_t init;

    /* Configure system clock */
    md_cmu_clock_config_default();
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(5);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();
    /* Set PCLK2 frequency */
    md_cmu_div_config(MD_CMU_PCLK_2, MD_CMU_DIV_2);

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* Initialize stdio */
    uart_stdio_init();
    /* Initialize RTC */
    md_rtc_struct_init(&init);
    md_rtc_init(&init);

    printf_e("System start...\r\n");
    /* Initialize time/date */
    md_rtc_time_struct_init(&time);
    md_rtc_date_struct_init(&date);
    md_rtc_set_time(&time, MD_RTC_FORMAT_DEC);
    md_rtc_set_date(&date, MD_RTC_FORMAT_DEC);
    printf_e("year month day hour minute second\r\n");
    printf_e("----------------------------------------\r\n");
    printf_e("20%d-%d-%d %d:%d:%d\r\n", date.year, date.month,
             date.day, time.hour, time.minute, time.second);
    /* Set current time and date */
    time.hour    = 23;
    time.minute  = 59;
    time.second  = 55;
    date.day     = 7;
    date.month   = 5;
    date.year    = 24;
    date.week    = 2;
    md_rtc_set_time(&time, MD_RTC_FORMAT_DEC);
    md_rtc_set_date(&date, MD_RTC_FORMAT_DEC);
    printf_e("20%d-%d-%d %d:%d:%d\r\n", date.year, date.month,
             date.day, time.hour, time.minute, time.second);

    while (1)
    {
        /* Get time/date each second */
        md_delay_1ms(1000);
        md_rtc_get_time(&time, MD_RTC_FORMAT_DEC);
        md_rtc_get_date(&date, MD_RTC_FORMAT_DEC);
        printf_e("20%d-%d-%d %d:%d:%d\r\n", date.year, date.month,
                 date.day, time.hour, time.minute, time.second);
    }
}

/**
  * @}
  */
/**
  * @}
  */
