/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */

#include <string.h>
#include "main.h"

uint8_t low_power_flag = 0;
/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/* Only one macro can be opened at a time */
#define LP_MODE_STOP1
//#define LP_MODE_STOP2
//#define LP_MODE_STANDBY

/**
  * @brief  Initializate pin.
  * @retval None
  */
void gpio_pin_init(void)
{
    md_gpio_init_t x;

    /* Initialize GPIO_PIN_15 as output */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_1;
    md_gpio_init(GPIOA, MD_GPIO_PIN_15, &x);
    MD_GPIOA_SET_PIN(MD_GPIO_PIN_15);

    /* Initialize GPIO_PIN_2 as input */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_1;
    md_gpio_init(GPIOB, MD_GPIO_PIN_2, &x);

}
void md_exti2_iqr_handler()
{
    if (EXTI->EXTIFLAG & 0x4)
    {
        EXTI->EXTICFR = 0x4;
    }
}
void md_lvd_iqr_handler()
{
    if (md_pmu_is_enable_lvd_it() && md_pmu_is_active_flag_lvd_it())
    {
        md_pmu_set_lvd_it_flag(); /* clear LVD interrupt flag */

#ifdef LP_MODE_STOP1
        md_pmu_set_low_power_mode(MD_PMU_LP_STOP1);
#endif
#ifdef LP_MODE_STOP2
        md_pmu_set_low_power_mode(MD_PMU_LP_STOP2);
#endif
#ifdef LP_MODE_STANDBY
        md_pmu_set_low_power_mode(MD_PMU_LP_STANDBY);
#endif
        SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
        SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
        __WFI();
        SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;
    }
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_clock_config_default();
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(5);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();
    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* Enable external interrupt */
    md_mcu_irq_config(EXTI2_IRQn, 3, 3, ENABLE);
    /* Initialize pin */
    gpio_pin_init();

    /*Sets port of the interrup*/
    md_gpio_set_interrupt_port(GPIOB, MD_GPIO_PIN_2);
    /*Enable filter*/
    md_gpio_interrupt_filter_enable(MD_GPIO_PIN_2);
    /*Select filter clock*/
    md_gpio_interrupt_filter_clock_select(EXTI_FILTER_CLOCK_10K);
    /*Set filter time */
    md_gpio_interrupt_filter_time_set(10);
    /* Clear interrupt flag */
    md_gpio_clear_flag(MD_GPIO_PIN_2);
    /* Configure trailing edge interrupt */
    md_gpio_enable_trailing_edge_trigger(MD_GPIO_PIN_2);
    md_gpio_enable_external_interrupt(MD_GPIO_PIN_2);

    md_pmu_lvd_config(MD_PMU_LVD_VOL_SEL_3_5, MD_PMU_LVD_TRIGGER_HIGH_LEVEL, ENABLE);

    /* Enable LVD interrupt */
    md_pmu_set_lvd_it_flag();
    md_mcu_irq_config(LVD_IRQn, 0, 3, ENABLE);
    md_pmu_enable_lvd_it();

    while (1)
    {
        MD_GPIOA_TOGGLE_PIN(MD_GPIO_PIN_15);
        md_delay_1ms(400);
    }
}

/**
  * @}
  */
/**
  * @}
  */
