/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for demo
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

md_uart_init_t h_uart;
md_pis_init_t p_init;

uint8_t rx_buf[8] = {0};
uint8_t tx_buf[8] = {0};
uint8_t rec_flag = 0;
uint8_t rec_cnt = 0;

/**
  * @brief  Init UART pin
  * @retval None
  */
void md_uart_pin_init(void)
{
    md_gpio_init_t x;

    /* Initialize rx pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_6;
    md_gpio_init(GPIOC, MD_GPIO_PIN_6, &x); /* PIS ch0 */

    /* Initialize rx pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_1;
    md_gpio_init(GPIOC, MD_GPIO_PIN_7, &x);
}

/**
  * @brief  Test main function
  * @retval Status.
  */int a = 0;
int main()
{
    uint8_t i = 0;

    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(2);
    md_mcu_irq_config(UART5_IRQn, 1, 1, ENABLE);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Initialize gpio*/
    md_uart_pin_init();
    md_uart_init_struct(&h_uart);
    md_uart_init(UART5, &h_uart);

    /* Initialize PIS */
    memset(&p_init, 0, sizeof(md_pis_init_t));
    /* Initialize PIS RX */
    p_init.p_src     = MD_PIS_GPIOC_PIN7;
    p_init.p_clk     = MD_PIS_CLK_PCLK1;
    p_init.p_edge    = MD_PIS_EDGE_NONE;
    p_init.p_output  = MD_PIS_OUT_LEVEL;
    p_init.c_trig    = MD_PIS_CH14_UART5_RXD;
    p_init.c_clk     = MD_PIS_CLK_PCLK1;
    md_pis_init(&p_init);
    md_uart_set_rx_fifo_reset(UART5);
    md_uart_enable_it_rfth(UART5);

    /* Initialize PIS TX */
    p_init.p_src     = MD_PIS_UART5_TXOUT;
    p_init.p_clk     = MD_PIS_CLK_PCLK1;
    p_init.p_edge    = MD_PIS_EDGE_NONE;
    p_init.p_output  = MD_PIS_OUT_LEVEL;
    p_init.c_trig    = MD_PIS_GPIO_CH0_OUTPUT;
    p_init.c_clk     = MD_PIS_CLK_PCLK1;
    md_pis_init(&p_init);

    memset(tx_buf, 0xAA, sizeof(tx_buf));

    while (1)
    {
        md_delay_1ms(1000);

        for (i = 0; i < 8; i++)
        {
            md_uart_set_send_data8(UART5, tx_buf[i]);

            while (!md_uart_is_trigger_fifo_threshold_tfth(UART5));
        }

        if (0x1 == rec_flag)
        {
            for (i = 0; i < 8; i++)
            {
                md_uart_set_send_data8(UART5, rx_buf[i]);

                while (!md_uart_is_trigger_fifo_threshold_tfth(UART5));
            }

            rec_flag = 0x0;
            rec_cnt = 0x0;
            md_uart_set_rx_fifo_reset(UART5);
        }
    }
}

