/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */
/** @addtogroup Public_Function
  * @{
  */

uint32_t nm_value;
md_adc_init_t h_adc;
md_pis_init_t p_init;
md_adc_nch_conf_t config;

/**
  * @brief  Initializate pin of adc module.
  * @retval None
  */
void adc_pin_init(void)
{
    md_gpio_init_t x;

    /* Initialize sck pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_0;
    md_gpio_init(GPIO_CH_PORT, ADC_CH0_PIN, &x);

    return;
}

/**
  * @brief  Initialize TIMER.
  * @param  None
  * @retval None
  */
void timer_init(void)
{
    md_timer_base_init_t init;

    md_timer_base_struct_init(&init);
    init.prescaler = (uint16_t)0x0003;
    init.period = 0xFFFF;
    md_timer_base_set_config(AD16C4T0, &init);
}

/**
  * @brief  Initialize PIS.
  * @param  None
  * @retval None
  */
void pis_init(void)
{
    /* Initialize PIS */
    memset(&p_init, 0, sizeof(md_pis_init_t));

    p_init.p_src     = MD_PIS_TIMER0_UPDATA;
    p_init.p_clk     = MD_PIS_CLK_PCLK1;
    p_init.p_edge    = MD_PIS_EDGE_NONE;
    p_init.p_output  = MD_PIS_OUT_PULSE;
    p_init.c_trig    = MD_PIS_CH6_ADC0_NORMAL;
    p_init.c_clk     = MD_PIS_CLK_PCLK2;

    md_pis_init(&p_init);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(5);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Initialize adc pin */
    adc_pin_init();

    /* Initialize ADC0 */
    md_adc_struct_init(&h_adc);
    md_adc_normal_struct_init(&config);
    md_adc_init(ADC0, &h_adc);
    md_adc_normal_channel_config(ADC0, &config);

    /* Enable ADC0 and start normal convert */
    md_adc_converter_enable_adcen(ADC0);
    /* Enable extarnal trigger */
    md_adc_set_normal_conv_extern_polarity(ADC0, MD_ADC_ETS_RISE);
    /* Initialize PIS */
    pis_init();
    md_adc_set_normal_channel_pis_trig_netp(ADC0, 6);
    /* Initialize TIMER */
    timer_init();

    /* Start TIMER */
    md_timer_enable_counter_cnten(AD16C4T0);

    /* Wait convert finish */
    while (md_adc_get_stat_nche(ADC0) == 0);

    /* Stop TIMER */
    md_timer_disable_counter_cnten(AD16C4T0);
    /* Clear ADC complete flag */
    md_adc_set_clr_nche(ADC0);
    /* Read normal convert result */
    nm_value = md_adc_get_normal_channel_val(ADC0);

    while (1)
    {
        /* Start TIMER */
        md_timer_enable_counter_cnten(AD16C4T0);

        /* Wait convert finish */
        while (md_adc_get_stat_nche(ADC0) == 0);

        /* Stop TIMER */
        md_timer_disable_counter_cnten(AD16C4T0);
        /* Clear ADC complete flag */
        md_adc_set_clr_nche(ADC0);
        /* Read normal convert result */
        nm_value = md_adc_get_normal_channel_val(ADC0);

        md_delay_1ms(10);
    }
}

/**
  * @}
  */
/**
  * @}
  */
