/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for demo
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

md_lpuart_handle_t h_uart;
md_lpuart_init_t lpuart_init;
uint8_t rtx_buf[8] = {0};
uint8_t rtx_flag = 0;
/**
  * @brief  Init LPUART pin
  * @retval None
  */
void md_lpuart_pin_init(void)
{
    md_gpio_init_t x;
    /* Initialize tx pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_2;
    md_gpio_init(GPIOA, MD_GPIO_PIN_5, &x);

    /* Initialize rx pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_2;
    md_gpio_init(GPIOA, MD_GPIO_PIN_4, &x);
}

/**
  * @brief  Handle the DMA interrupt,.
  * @param  UARTx: Pointer to UART peripheral.
  * @retval None
  */
void md_dma0_irq_handler(LPUART_TypeDef *LPUARTx)
{
    uint32_t i = 0, reg;

    reg = DMA0->IFM;

    // Transfer completely ch0
    if (reg & (0x01))
    {
        h_uart.tx_count = h_uart.tx_size;
        md_lpuart_disable_txdma(LPUARTx);
        h_uart.tx_count = 0;
        WRITE_REG(DMA0->ICR, 0x01);
        rtx_flag = 0;
    }

    // Receiver completely ch1
    if (reg & (1U << 2 * 1))
    {
        h_uart.rx_count = h_uart.rx_size;
        md_lpuart_disable_rxdma(LPUARTx);
        h_uart.rx_count = 0;
        rtx_flag = 1;
        WRITE_REG(DMA0->ICR, (1U << (2U * 1)));
    }

    if (reg & (1U << (2U * i + 1)))
    {
        WRITE_REG(DMA0->ICR, (1U << (2U * i + 1U)));
    }
}
/**
  * @brief  Sends an amount of data in non blocking mode.
  * @param  LPUARTx: Pointer to LPUART peripheral.
  * @param  DMAx: Pointer to DMA peripheral
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @param  channel: DMA channel as UART transmit
  * @retval Status, see @ref md_status_t.
  */
md_status_t md_lpuart_send_by_dma(LPUART_TypeDef *LPUARTx, DMA_TypeDef *DMA, uint8_t *buf, uint16_t size, uint8_t channel)
{
    md_dma_config_t dma_config;
    md_dma_reset(DMA);
    md_dma_enable_it_tc(DMA, channel);

    memset(&dma_config, 0x0, sizeof(md_dma_config_t));
    /* Set the dma parameters */
    dma_config.src            = (void *)buf;
    dma_config.dst            = (void *)&LPUARTx->TXDR;
    dma_config.size           = size;
    dma_config.src_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.dst_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.src_inc        = MD_DMA_DATA_INC_ENABLE;
    dma_config.dst_inc        = MD_DMA_DATA_INC_DISABLE;
    dma_config.R_power        = MD_DMA_R_POWER_1;
    dma_config.priority       = MD_DMA_LOW_PRIORITY;
    dma_config.mem_to_mem     = DISABLE;
    dma_config.circle_mode    = DISABLE;
    dma_config.msel           = MD_DMA_MSEL_LPUART0;
    dma_config.msigsel        = MD_DMA_MSIGSEL_LPUART_TXEMPTY;
    dma_config.dir            = MD_DMA_DIR_TO_PERH;
    md_dma_init(DMA, channel, &dma_config);
    md_lpuart_enable_txdma(LPUARTx);

    return MD_OK;
}
/**
  * @brief  Receives an amount of data in non blocking mode.
  * @param  LPUARTx: Pointer to LPUART peripheral.
  * @param  DMAx: Pointer to DMA peripheral
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @param  channel: DMA channel as UART receive
  * @retval Status, see @ref md_status_t.
  */
md_status_t md_lpuart_recv_by_dma(LPUART_TypeDef *LPUARTx, DMA_TypeDef *DMA, uint8_t *buf, uint16_t size, uint8_t channel)
{
    md_dma_config_t dma_config;

    md_dma_reset(DMA);
    md_dma_enable_it_tc(DMA, channel);
    memset(&dma_config, 0x0, sizeof(md_dma_config_t));
    /* Set the dma parameters */
    dma_config.src            = (void *)&LPUARTx->RXDR;
    dma_config.dst            = (void *)buf;
    dma_config.size           = size;
    dma_config.src_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.dst_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.src_inc        = MD_DMA_DATA_INC_DISABLE;
    dma_config.dst_inc        = MD_DMA_DATA_INC_ENABLE;
    dma_config.R_power        = MD_DMA_R_POWER_1;
    dma_config.priority       = MD_DMA_LOW_PRIORITY;
    dma_config.mem_to_mem     = DISABLE;
    dma_config.circle_mode    = DISABLE;
    dma_config.msel           = MD_DMA_MSEL_LPUART0;
    dma_config.msigsel        = MD_DMA_MSIGSEL_LPUART_RNR;
    dma_config.dir            = MD_DMA_DIR_TO_SRAM;
    md_dma_init(DMA, channel, &dma_config);
    md_lpuart_enable_rxdma(LPUARTx);

    return MD_OK;
}
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(2);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Initialize uart communication gpio*/
    md_lpuart_pin_init();
    md_lpuart_init_struct(&lpuart_init);
    md_lpuart_init(LPUART0, &lpuart_init);
    md_mcu_irq_config(LPUART0_IRQn, 1, 1, ENABLE);
    md_mcu_irq_config(DMA0_IRQn, 2, 2, ENABLE);
    memset(rtx_buf, 0x55, sizeof(rtx_buf));

    while (1)
    {
        /* Receive a message */
        md_lpuart_recv_by_dma(LPUART0, DMA0, rtx_buf, 8, 1);
        md_delay_1ms(500);

        if (rtx_flag)
        {
            /* Send a message */
            md_lpuart_send_by_dma(LPUART0, DMA0, rtx_buf, 8, 0);
            md_delay_1ms(500);
        }


    }
}

