/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for demo
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

md_lpuart_handle_t h_uart;
md_lpuart_init_t lpuart_init;
uint8_t rtx_buf[8] = {0};

/**
  * @brief  Init LPUART pin
  * @retval None
  */
void md_lpuart_pin_init(void)
{
    md_gpio_init_t x;
    /* Initialize tx pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_2;
    md_gpio_init(GPIOA, MD_GPIO_PIN_5, &x);

    /* Initialize rx pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_2;
    md_gpio_init(GPIOA, MD_GPIO_PIN_4, &x);
}
/**
  * @brief  This function handles LPUART interrupt request.
  * @param  hperh: Pointer to a ald_lpuart_handle_t structure that contains
  *         the configuration information for the specified LPUART module.
  * @retval None
  */

void md_lpuart_irq_handler(LPUART_TypeDef *LPUARTx)
{
    uint32_t flag;
    uint32_t source;

    uint8_t tmp;
    uint16_t i;

    /* Receive */
    flag   = md_lpuart_is_active_flag_rbr(LPUARTx);
    source = md_lpuart_is_enabled_it_rbr(LPUARTx);;

    if ((flag != RESET) && (source != RESET))
    {
        do
        {
            i   = 10000;
            tmp = md_lpuart_get_state_rxptr(LPUARTx);
            *h_uart.rx_buf++ = LPUARTx->RXDR;
            --h_uart.rx_count;

            while ((md_lpuart_get_state_rxptr(LPUARTx) != tmp - 1) && (i--));
        }
        while (md_lpuart_get_state_rxptr(LPUARTx));

        if (h_uart.rx_count == 0)
        {
            md_lpuart_disable_it_rbr(LPUARTx);
            md_lpuart_disable_it_perr(LPUARTx);
            md_lpuart_disable_it_ferr(LPUARTx);

            /*send data of received */
            h_uart.tx_count = 8;
            h_uart.tx_buf = rtx_buf;
            h_uart.tx_size = 8;
            md_lpuart_enable_it_tbemp(LPUARTx);
        }
    }

    /* Transmite */
    flag   = md_lpuart_is_active_flag_tbemp(LPUARTx);
    source = md_lpuart_is_enabled_it_tbemp(LPUARTx);

    if ((flag != RESET) && (source != RESET))
    {
        LPUART0->TXDR = (uint8_t)(*h_uart.tx_buf++);

        if (--h_uart.tx_count == 0)
        {
            md_lpuart_disable_it_tbemp(LPUARTx);
            md_lpuart_enable_it_tc(LPUARTx);
        }
    }

    /* End Transmite */
    flag   = md_lpuart_is_active_flag_tc(LPUARTx);
    source = md_lpuart_is_enabled_it_tc(LPUARTx);

    if ((flag != RESET) && (source != RESET))
    {
        md_lpuart_disable_it_tc(LPUARTx);

        /* receive init*/
        h_uart.rx_count = 8;
        h_uart.rx_buf = rtx_buf;
        h_uart.rx_size = 8;
        md_lpuart_enable_it_rbr(LPUART0);
    }
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{

    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(5);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Initialize uart communication gpio*/
    md_lpuart_pin_init();
    md_lpuart_init_struct(&lpuart_init);
    md_lpuart_init(LPUART0, &lpuart_init);
    md_mcu_irq_config(LPUART0_IRQn, 3, 3, ENABLE);
    memset(rtx_buf, 0x55, sizeof(rtx_buf));

    /* receive init*/
    h_uart.rx_count = 8;
    h_uart.rx_buf = rtx_buf;
    h_uart.rx_size = 8;
    md_lpuart_enable_it_rbr(LPUART0);

    while (1)
    {
        md_delay_1ms(1000);

    }
}

