/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

md_i2c_init_t init;
/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
uint8_t g_recv_buf[8];
uint8_t g_recv_len = sizeof(g_recv_buf);
uint8_t g_recv_inc = 0U;
uint8_t g_send_buf[8];
uint8_t g_send_len = sizeof(g_send_buf);
uint8_t g_send_inc = 0U;
uint8_t g_complete = 1U;
uint8_t g_error_flag = 0U;
uint8_t g_rec_flag = 0;
/**
  * @brief  Initializate pin of i2c module.
  * @retval None
  */
void i2c_pin_init(void)
{
    md_gpio_init_t x;

    /* Initialize scl pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_OPEN_DRAIN;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_5;
    md_gpio_init(GPIOB, MD_GPIO_PIN_10, &x);

    /* Initialize sda pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_OPEN_DRAIN;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_5;
    md_gpio_init(GPIOB, MD_GPIO_PIN_11, &x);

    return;
}

/**
 * @brief  Transmit in master mode an amount of data in non-blocking mode with Interrupt
 * @param  hperh: Pointer to a ald_i2c_handle_t structure that contains
 *                the configuration information for the specified I2C.
 * @param  dev_addr: Target device address
 * @param  buf: Pointer to data buffer
 * @param  size: Amount of data to be sent
  * @retval Status, see @ref ald_status_t.
 */
void dma_send_tx(void)
{
    md_dma_config_t dma_config;

    while (md_i2c_is_active_busy(I2C1));

    md_dma_enable_it_tc(DMA0, MD_DMA_CH_0);

    memset(&dma_config, 0x0, sizeof(md_dma_config_t));
    /* Set the dma parameters */
    dma_config.src            = (void *)g_send_buf;
    dma_config.dst            = (void *)&I2C1->TXDATA;
    dma_config.size           = g_send_len;
    dma_config.src_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.dst_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.src_inc        = MD_DMA_DATA_INC_ENABLE;
    dma_config.dst_inc        = MD_DMA_DATA_INC_DISABLE;
    dma_config.R_power        = MD_DMA_R_POWER_1;
    dma_config.priority       = MD_DMA_LOW_PRIORITY;
    dma_config.mem_to_mem     = DISABLE;
    dma_config.circle_mode    = DISABLE;
    dma_config.msel           = MD_DMA_MSEL_I2C1;
    dma_config.msigsel        = MD_DMA_MSIGSEL_I2C_TXEMPTY;
    dma_config.dir            = MD_DMA_DIR_TO_PERH;
    md_dma_init(DMA0, MD_DMA_CH_0, &dma_config);
    md_dma_set_channel_single(DMA0, MD_DMA_CH_0);
}

/**
  * @brief  Receive in master mode an amount of data in non-blocking mode with Interrupt
  * @param  hperh: Pointer to a ald_i2c_handle_t structure that contains
  *                the configuration information for the specified I2C.
  * @param  dev_addr: Target device address
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref ald_status_t.
  */
void dma_recv_rx(void)
{
    md_dma_config_t dma_config;

    while (md_i2c_is_active_busy(I2C1));

    md_dma_enable_it_tc(DMA0, MD_DMA_CH_1);

    memset(&dma_config, 0x0, sizeof(md_dma_config_t));
    /* Set the dma parameters */
    dma_config.src            = (void *)&I2C1->RXDATA;
    dma_config.dst            = (void *)g_recv_buf;
    dma_config.size           = g_recv_len;
    dma_config.src_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.dst_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.src_inc        = MD_DMA_DATA_INC_DISABLE;
    dma_config.dst_inc        = MD_DMA_DATA_INC_ENABLE;
    dma_config.R_power        = MD_DMA_R_POWER_1;
    dma_config.priority       = MD_DMA_LOW_PRIORITY;
    dma_config.mem_to_mem     = DISABLE;
    dma_config.circle_mode    = DISABLE;
    dma_config.msel           = MD_DMA_MSEL_I2C1;
    dma_config.msigsel        = MD_DMA_MSIGSEL_I2C_RNR;
    dma_config.dir            = MD_DMA_DIR_TO_SRAM;
    md_dma_init(DMA0, MD_DMA_CH_1, &dma_config);
    md_dma_set_channel_single(DMA0, MD_DMA_CH_1);
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    uint16_t cnt;
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(2);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    i2c_pin_init();
    memset(&init, 0, sizeof(md_i2c_init_t));
    md_i2c_struct_init(&init);
    /* Initialize I2C */
    init.module       = MD_I2C_MODULE_MASTER;
    init.addr_mode    = MD_I2C_ADDR_7BIT;
    init.clk_speed    = 100000;
    init.dual_addr    = MD_I2C_DUALADDR_ENABLE;
    init.general_call = MD_I2C_GENERALCALL_ENABLE;
    init.no_stretch   = MD_I2C_NOSTRETCH_ENABLE;
    init.own_addr1    = 0xA1;
    md_i2c_init(I2C1, &init);

    md_mcu_irq_config(I2C1_EV_IRQn, 1, 1, ENABLE);
    md_mcu_irq_config(DMA0_IRQn, 1, 0, ENABLE);

    /* set slave address */
    md_i2c_set_slave_addr(I2C1, SLAVE_ADDR);
    /* set transmission length */
    md_i2c_set_transmit_length(I2C1, g_send_len);
    /* Enable NACK interrupt */
    md_i2c_enable_it_nack(I2C1);
    /* DMA init */
    dma_recv_rx();
    dma_send_tx();

    while (1)
    {
        if (g_rec_flag == 0x1)
        {
            g_rec_flag = 0x0;

            memcpy(g_send_buf, g_recv_buf, g_recv_len);

            /* Enable DMA channel */
            md_dma_enable_channel(DMA0, MD_DMA_CH_0);
            /* select master write mode */
            md_i2c_enable_master_write(I2C1);
            /* enable TX DMA */
            md_i2c_enable_dma_transmit(I2C1);
            /* produce start signal */
            md_i2c_set_start(I2C1);

            cnt = 40000;

            while (!g_complete && (--cnt));

            cnt = 40000;

            while (md_i2c_is_active_busy(I2C1) && (--cnt));
        }

        /* receive */
        g_complete = 0;
        /* Enable DMA channel */
        md_dma_enable_channel(DMA0, MD_DMA_CH_1);
        /* select master read mode */
        md_i2c_enable_master_read(I2C1);
        /* enable RX DMA */
        md_i2c_enable_dma_receive(I2C1);
        /* produce start signal */
        md_i2c_set_start(I2C1);

        cnt = 40000;

        while (!g_complete && (--cnt));

        cnt = 40000;

        while (md_i2c_is_active_busy(I2C1) && (--cnt));

        md_delay_1ms(2000);
    }
}

/**
  * @}
  */
/**
  * @}
  */
