/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

md_i2c_init_t init;
md_i2c_handle_t h_i2c;
/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
uint8_t rx_buf[8];
uint8_t tx_buf[8];
/**
  * @brief  Initializate pin of i2c module.
  * @retval None
  */
void i2c_pin_init(void)
{
    md_gpio_init_t x;

    /* Initialize scl pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_OPEN_DRAIN;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_5;
    md_gpio_init(GPIOB, MD_GPIO_PIN_10, &x);

    /* Initialize sda pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_OPEN_DRAIN;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_5;
    md_gpio_init(GPIOB, MD_GPIO_PIN_11, &x);

    return;
}
/**
  * @brief  This function handles I2C event interrupt request.
  * @param  hperh: Pointer to a ald_i2c_handle_t structure that contains
  *                the configuration information for the specified I2C.
  * @retval None
  */
void md_i2c_ev_irq_handler(I2C_TypeDef *I2Cx)
{
    uint32_t size = h_i2c.xfer_size - h_i2c.xfer_count;

    /**< Transmit empty interrupt */
    if (md_i2c_mask_it_txe(I2Cx))
    {
        if (init.module == MD_I2C_MODULE_MASTER)
        {
            if (h_i2c.xfer_count != h_i2c.xfer_size)
            {
                I2Cx->TXDATA = (*h_i2c.p_buff++);
                h_i2c.xfer_count++;
            }
        }
        else if (init.module == MD_I2C_MODULE_SLAVE)
        {
            if (h_i2c.xfer_size > h_i2c.xfer_count)
            {
                I2Cx->TXDATA = (*h_i2c.p_buff++);
                h_i2c.xfer_count++;
            }
        }

        md_i2c_clear_flag_txe(I2Cx);
    }

    /**< Receive not empty interrupt */
    if (md_i2c_mask_it_rxne(I2Cx))
    {
        if (init.module == MD_I2C_MODULE_MASTER)
        {
            if ((h_i2c.xfer_size - h_i2c.xfer_count) > 0)
            {
                (*h_i2c.p_buff++) = I2Cx->RXDATA;
                h_i2c.xfer_count++;
            }
        }
        else if (init.module == MD_I2C_MODULE_SLAVE)
        {
            if (h_i2c.xfer_size > h_i2c.xfer_count)
            {
                (*h_i2c.p_buff++) = I2Cx->RXDATA;
                h_i2c.xfer_count++;
            }
        }

        md_i2c_clear_flag_rxne(I2Cx);
    }

    /**< Transmit completed interrupt */
    if (md_i2c_mask_it_tc(I2Cx))
    {
        if (init.module == MD_I2C_MODULE_MASTER)
        {
            if (md_i2c_is_enable_master_read(I2Cx))
            {
                md_i2c_disable_it_rxne(I2Cx);
                md_i2c_disable_it_tc(I2Cx);
                md_i2c_disable_it_tcr(I2Cx);

                if (RESET == md_i2c_is_enable_auto_end(I2Cx))
                {
                    md_i2c_set_stop(I2Cx);
                }
            }
            else
            {
                md_i2c_disable_it_txe(I2Cx);
                md_i2c_disable_it_nack(I2Cx);
                md_i2c_disable_it_tc(I2Cx);
                md_i2c_disable_it_tcr(I2Cx);

                if (RESET == md_i2c_is_enable_auto_end(I2Cx))
                {
                    md_i2c_set_stop(I2Cx);
                }
            }
        }

        md_i2c_clear_flag_tc(I2Cx);
    }

    /**< Transmit and reload completed interrupt */
    if (md_i2c_mask_it_tcr(I2Cx))
    {
        if (size > 0xFFFF)
        {
            MODIFY_REG(I2Cx->CON2, I2C_CON2_NBYTESL_MSK, 0xFFU << I2C_CON2_NBYTESL_POSS);
            MODIFY_REG(I2Cx->CON1, I2C_CON1_NBYTESH_MSK, 0xFFU << I2C_CON1_NBYTESH_POSS);
        }
        else
        {
            MODIFY_REG(I2Cx->CON2, I2C_CON2_NBYTESL_MSK, (size & 0xFF) << I2C_CON2_NBYTESL_POSS);
            MODIFY_REG(I2Cx->CON1, I2C_CON1_NBYTESH_MSK, (size >> 8) << I2C_CON1_NBYTESH_POSS);
            CLEAR_BIT(I2Cx->CON2, I2C_CON2_RELOAD_MSK);
        }

        md_i2c_clear_flag_tcr(I2Cx);
    }

    /**< Address matching interrupt */
    if (md_i2c_mask_it_addr(I2Cx))
        md_i2c_clear_flag_addr(I2Cx);

    /**< Stop detection interrupt */
    if (md_i2c_mask_it_stop(I2Cx))
    {
        md_i2c_disable_it_addr(I2Cx);
        md_i2c_disable_it_nack(I2Cx);
        md_i2c_disable_it_rxne(I2Cx);
        md_i2c_disable_it_txe(I2Cx);
    }
}
/**
  * @brief  This function handles I2C error interrupt request.
  * @param  hperh: pointer to a ald_i2c_handle_t structure that contains
  *         the configuration information for I2C module
  * @retval  NONE
  */
void md_i2c_er_irq_handler(I2C_TypeDef *I2Cx)
{
    /**< Transmit overrun interrupt */
    if (md_i2c_mask_it_txov(I2Cx))
    {
        md_i2c_clear_flag_txov(I2Cx);
    }

    /**< Transmit underrun */
    if (md_i2c_mask_it_txud(I2Cx))
    {
        md_i2c_clear_flag_txud(I2Cx);
    }

    /**< Receive overrun interrupt */
    if (md_i2c_mask_it_rxov(I2Cx))
    {
        md_i2c_clear_flag_rxov(I2Cx);
    }

    /**< Receive underrun interrupt */
    if (md_i2c_mask_it_rxud(I2Cx))
    {
        md_i2c_clear_flag_rxud(I2Cx);
    }

    /**< NACK interrupt */
    if (md_i2c_mask_it_nack(I2Cx))
    {
        if (h_i2c.xfer_count == h_i2c.xfer_size)
        {
            md_i2c_disable_it_nack(I2Cx);
            md_i2c_disable_it_txe(I2Cx);
            md_i2c_clear_flag_nack(I2Cx);
            return;
        }

        if (init.module == MD_I2C_MODULE_MASTER)
            md_i2c_set_stop(I2Cx);

        md_i2c_disable_it_nack(I2Cx);
        md_i2c_disable_it_txe(I2Cx);
        md_i2c_clear_flag_nack(I2Cx);
    }

    /**< Bus error interrupt */
    if (md_i2c_mask_it_berr(I2Cx))
    {
        md_i2c_clear_flag_berr(I2Cx);
    }

    /**< Arbitration loss interrupt */
    if (md_i2c_mask_it_arlo(I2Cx))
    {
        md_i2c_clear_flag_arlo(I2Cx);
    }

    /**< PEC error interrupt */
    if (md_i2c_mask_it_pece(I2Cx))
    {
        md_i2c_clear_flag_pece(I2Cx);
    }

    /**< timeout interrupt */
    if (md_i2c_mask_it_tout(I2Cx))
    {
        md_i2c_clear_flag_tout(I2Cx);
    }

    /**< SMBus Alert interrupt */
    if (md_i2c_mask_it_alert(I2Cx))
    {
        md_i2c_clear_flag_alert(I2Cx);
    }
}

/**
 * @brief  Transmit in master mode an amount of data in non-blocking mode with Interrupt
 * @param  hperh: Pointer to a ald_i2c_handle_t structure that contains
 *                the configuration information for the specified I2C.
 * @param  dev_addr: Target device address
 * @param  buf: Pointer to data buffer
 * @param  size: Amount of data to be sent
  * @retval Status, see @ref ald_status_t.
 */
md_status_t md_i2c_master_send_by_it(I2C_TypeDef *I2Cx, uint8_t *buf, uint32_t size)
{
    while (md_i2c_is_active_busy(I2Cx));

    h_i2c.p_buff     = buf;
    h_i2c.xfer_size  = size;
    h_i2c.xfer_count = 0;
    md_i2c_enable_master_write(I2Cx);

    if (size <= 0xFFFF)
    {
        MODIFY_REG(I2Cx->CON2, I2C_CON2_NBYTESL_MSK, (size & 0xFF) << I2C_CON2_NBYTESL_POSS);
        MODIFY_REG(I2Cx->CON1, I2C_CON1_NBYTESH_MSK, (size >> 8) << I2C_CON1_NBYTESH_POSS);
    }
    else
    {
        MODIFY_REG(I2Cx->CON2, I2C_CON2_NBYTESL_MSK, 0xFFU << I2C_CON2_NBYTESL_POSS);
        MODIFY_REG(I2Cx->CON1, I2C_CON1_NBYTESH_MSK, 0xFFU << I2C_CON1_NBYTESH_POSS);
        SET_BIT(I2Cx->CON2, I2C_CON2_RELOAD_MSK);
    }

    md_i2c_clear_flag_txe(I2Cx);
    md_i2c_clear_flag_nack(I2Cx);
    md_i2c_clear_flag_tc(I2Cx);
    md_i2c_clear_flag_tcr(I2Cx);
    md_i2c_set_start(I2Cx);

    md_i2c_enable_it_txe(I2Cx);
    md_i2c_enable_it_nack(I2Cx);
    md_i2c_enable_it_tc(I2Cx);
    md_i2c_enable_it_tcr(I2Cx);
    return MD_OK;
}

/**
  * @brief  Receive in master mode an amount of data in non-blocking mode with Interrupt
  * @param  hperh: Pointer to a ald_i2c_handle_t structure that contains
  *                the configuration information for the specified I2C.
  * @param  dev_addr: Target device address
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref ald_status_t.
  */
md_status_t md_i2c_master_recv_by_it(I2C_TypeDef *I2Cx, uint8_t *buf, uint32_t size)
{

    h_i2c.p_buff     = buf;
    h_i2c.xfer_size  = size;
    h_i2c.xfer_count = 0;

    md_i2c_enable_master_read(I2Cx);

    if (size <= 0xFF)
    {
        MODIFY_REG(I2Cx->CON2, I2C_CON2_NBYTESL_MSK, (size & 0xFF) << I2C_CON2_NBYTESL_POSS);
        MODIFY_REG(I2Cx->CON1, I2C_CON1_NBYTESH_MSK, (size >> 8) << I2C_CON1_NBYTESH_POSS);
    }
    else
    {
        MODIFY_REG(I2Cx->CON2, I2C_CON2_NBYTESL_MSK, 0xFFU << I2C_CON2_NBYTESL_POSS);
        MODIFY_REG(I2Cx->CON1, I2C_CON1_NBYTESH_MSK, 0xFFU << I2C_CON1_NBYTESH_POSS);
        SET_BIT(I2Cx->CON2, I2C_CON2_RELOAD_MSK);
    }

    md_i2c_clear_flag_rxne(I2Cx);
    md_i2c_clear_flag_tc(I2Cx);
    md_i2c_clear_flag_tcr(I2Cx);
    md_i2c_enable_it_rxne(I2Cx);
    md_i2c_enable_it_tc(I2Cx);
    md_i2c_enable_it_tcr(I2Cx);
    md_i2c_set_start(I2Cx);
    return MD_OK;
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(5);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    memset(tx_buf, 0x55, sizeof(tx_buf));
    memset(rx_buf, 0x55, sizeof(rx_buf));
    i2c_pin_init();
    memset(&init, 0, sizeof(md_i2c_init_t));
    md_i2c_struct_init(&init);
    /* Initialize I2C */
    init.module       = MD_I2C_MODULE_MASTER;
    init.addr_mode    = MD_I2C_ADDR_7BIT;
    init.clk_speed    = 100000;
    init.dual_addr    = MD_I2C_DUALADDR_DISABLE;
    init.general_call = MD_I2C_GENERALCALL_ENABLE;
    init.no_stretch   = MD_I2C_NOSTRETCH_ENABLE;
    init.own_addr1    = 0xA1;
    md_i2c_init(I2C1, &init);

    NVIC_SetPriority(I2C1_EV_IRQn, 3);
    NVIC_EnableIRQ(I2C1_EV_IRQn);
    NVIC_SetPriority(I2C1_ERR_IRQn, 3);
    NVIC_EnableIRQ(I2C1_ERR_IRQn);
    /* set slave address */
    md_i2c_set_slave_addr(I2C1, SLAVE_ADDR);

    md_i2c_clear_flag_txe(I2C1);
    md_i2c_clear_flag_rxne(I2C1);

    while (1)
    {
        /* Send and Recv data by interrupt */
        md_i2c_master_send_by_it(I2C1, tx_buf, 8);
        md_delay_1ms(1000);
        md_i2c_master_recv_by_it(I2C1, rx_buf, 8);
        md_delay_1ms(3000);
    }
}

/**
  * @}
  */
/**
  * @}
  */
