/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

md_i2c_init_t init;

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
uint8_t tx_buf[8];
/**
  * @brief  Initializate pin of i2c module.
  * @retval None
  */
void i2c_pin_init(void)
{
    md_gpio_init_t x;

    /* Initialize scl pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_OPEN_DRAIN;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_5;
    md_gpio_init(GPIOB, MD_GPIO_PIN_10, &x);

    /* Initialize sda pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_OPEN_DRAIN;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_5;
    md_gpio_init(GPIOB, MD_GPIO_PIN_11, &x);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    uint16_t i;
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(2);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    memset(tx_buf, 0x55, sizeof(tx_buf));

    i2c_pin_init();
    memset(&init, 0, sizeof(md_i2c_init_t));
    md_i2c_struct_init(&init);
    /* Initialize I2C */
    init.module       = MD_I2C_MODULE_MASTER;
    init.addr_mode    = MD_I2C_ADDR_7BIT;
    init.clk_speed    = 100000;
    init.dual_addr    = MD_I2C_DUALADDR_DISABLE;
    init.general_call = MD_I2C_GENERALCALL_ENABLE;
    init.no_stretch   = MD_I2C_NOSTRETCH_ENABLE;
    init.own_addr1    = 0xA1;
    md_i2c_init(I2C1, &init);

    /* set slave address */
    md_i2c_set_slave_addr(I2C1, SLAVE_ADDR);
    md_i2c_clear_flag_txe(I2C1);
    md_i2c_clear_flag_rxne(I2C1);

    while (1)
    {
        /* set transmission length */
        md_i2c_set_transmit_length(I2C1, sizeof(tx_buf));
        /* select master write mode */
        md_i2c_enable_master_write(I2C1);
        /* produce start signal */
        md_i2c_set_start(I2C1);

        for (i = 0; i < sizeof(tx_buf); ++i)
        {
            md_i2c_set_tx_reg_data(I2C1, tx_buf[i]);

            while (!md_i2c_is_active_txe(I2C1));
        }

        md_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */
