/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

md_crc_init_t init;
uint32_t result = 0;
/* test case 0
 * CRC-8        refin false
 *      refout false
 *      init 0x00
 * input: 0x01 0x02
 * expect output: 0x1b
 *
 */
void test_crc8_1(void)
{
    uint8_t i;
    uint8_t crc_buf[2] = {1, 2};
    /* Clear md_crc_init_t structure */
    memset(&init, 0x0, sizeof(md_crc_init_t));
    /* Initialize CRC */
    init.mode = MD_CRC_MODE_8;
    init.seed = 0x0;
    init.data_rev = DISABLE;
    init.data_inv = DISABLE;
    init.chs_rev = DISABLE;
    init.chs_inv = DISABLE;
    init.len = MD_CRC_DATASIZE_8;
    md_crc_init(&init);

    /* start calculate and result is 0x1b*/
    for (i = 0; i < 2; i++)
        md_crc_write_data(CRC, crc_buf[i]);

    result = md_crc_get_check_result(CRC);
}
/* test
 * CRC-16/MODBUS refin  true
 *       refout true
 *       init   0x0000
 * input: 0x0001 0x0002
 * expect output: 0x25d0
 *
 */
void test_crc16_1(void)
{
    uint8_t i;
    uint8_t crc_buf[4] = {0, 1, 0, 2};
    /* Clear md_crc_init_t structure */
    memset(&init, 0x0, sizeof(md_crc_init_t));
    /* Initialize CRC */
    init.mode = MD_CRC_MODE_16;
    init.seed = 0xffff;
    init.data_rev = ENABLE;
    init.data_inv = DISABLE;
    init.chs_rev = ENABLE;
    init.chs_inv = DISABLE;
    init.len = MD_CRC_DATASIZE_8;
    md_crc_init(&init);

    /* start calculate and result is 0x25d0*/
    for (i = 0; i < 4; i++)
        md_crc_write_data(CRC, crc_buf[i]);

    result = md_crc_get_check_result(CRC);
}
/* test
 * crc32    refin false
 *      refout false
 *      init 0xffffffff
 * input: 0x01020304 0x05060708
 * expect output: 0x140B8DD8
 *
 */
void test_crc32_1(void)
{
    uint8_t i;
    uint8_t crc_buf[8] = {1, 2, 3, 4, 5, 6, 7, 8};
    /* Clear md_crc_init_t structure */
    memset(&init, 0x0, sizeof(md_crc_init_t));
    /* Initialize CRC */
    init.mode = MD_CRC_MODE_32;
    init.seed = 0xffffffff;
    init.data_rev = DISABLE;
    init.data_inv = DISABLE;
    init.chs_rev = DISABLE;
    init.chs_inv = DISABLE;
    init.len = MD_CRC_DATASIZE_8;
    md_crc_init(&init);

    /* start calculate and result is 0x140B8DD8*/
    for (i = 0; i < 8; i++)
        md_crc_write_data(CRC, crc_buf[i]);

    result = md_crc_get_check_result(CRC);
}
/* test
 * CRC-16/CCITT-FALSE refin  false
 *            refout false
 *            init   0x0000
 * input: 0x0001 0x0002
 * expect output: 0x93b2
 *
 */
void test_crc_ccitt(void)
{
    uint8_t i;
    uint8_t crc_buf[4] = {0, 1, 0, 2};
    /* Clear md_crc_init_t structure */
    memset(&init, 0x0, sizeof(md_crc_init_t));
    /* Initialize CRC */
    init.mode = MD_CRC_MODE_CCITT;
    init.seed = 0xffff;
    init.data_rev = DISABLE;
    init.data_inv = DISABLE;
    init.chs_rev = DISABLE;
    init.chs_inv = DISABLE;
    init.len = MD_CRC_DATASIZE_8;
    md_crc_init(&init);

    /* start calculate and result is 0x93b2*/
    for (i = 0; i < 4; i++)
        md_crc_write_data(CRC, crc_buf[i]);

    result = md_crc_get_check_result(CRC);
}
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_clock_config_default();
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(5);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* test  */
    test_crc8_1();
    test_crc16_1();
    test_crc32_1();
    test_crc_ccitt();

    while (1)
    {
        md_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */
