/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

uint32_t nm_value;
md_adc_init_t h_adc;
md_adc_nch_conf_t n_config;
md_pis_init_t p_init;
md_timer_base_init_t init;

/**
  * @brief  Initializate pin of adc module.
  * @retval None
  */
void adc_pin_init(void)
{
    md_gpio_init_t x;

    /* Initialize sck pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_0;
    md_gpio_init(GPIO_CH0_PORT, ADC_CH0_PIN, &x);
}

/** @addtogroup Public_Function
  * @{
  */
/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* PriorityGroup_2 */
    NVIC_SetPriorityGrouping(2);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Initialize adc pin */
    adc_pin_init();

    /* Initialize ADC0 */
    md_adc_struct_init(&h_adc);
    md_adc_normal_struct_init(&n_config);
    md_adc_init(ADC0, &h_adc);
    md_adc_normal_channel_config(ADC0, &n_config);
    md_adc_set_normal_conv_extern_polarity(ADC0, MD_ADC_ETS_RISE);
    MD_ADC_NCH_PIS_CHANNEL_CONFIG(ADC0, 6);

    /* Timer init */
    init.prescaler = 479;
    init.mode      = MD_TIMER_CNT_MODE_UP;
    init.period    = 99999;
    init.re_cnt    = 0;
    md_timer_base_set_config(GP32C4T0, &init);
    md_timer_enable_counter_cnten(GP32C4T0);
    md_timer_master_trgout_sel_trgosel(GP32C4T0, MD_TIMER_TRGO_UPDATE);

    /* PIS config */
    memset(&p_init, 0, sizeof(md_pis_init_t));
    p_init.p_src     = MD_PIS_TIMER2_TRGOUT;
    p_init.p_clk     = MD_PIS_CLK_PCLK1;
    p_init.p_edge    = MD_PIS_EDGE_NONE;
    p_init.p_output  = MD_PIS_OUT_PULSE;
    p_init.c_trig    = MD_PIS_CH6_ADC0_NORMAL;
    p_init.c_clk     = MD_PIS_CLK_PCLK2;
    md_pis_init(&p_init);

    /* Start normal convert */
    while (md_adc_get_stat_nche(ADC0) == 0);

    md_adc_set_clr_nche(ADC0);
    nm_value = md_adc_get_normal_channel_val(ADC0);

    while (1)
    {
        while (md_adc_get_stat_nche(ADC0) == 0);

        md_adc_set_clr_nche(ADC0);
        nm_value = md_adc_get_normal_channel_val(ADC0);
    }
}

/**
  * @}
  */
/**
  * @}
  */
