/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

ald_uart_handle_t h_uart;

uint16_t tx_buf[8];
uint16_t rx_buf[8];

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void uart_pin_init(void)
{
    ald_gpio_init_t x;

    /* Initialize tx pin */
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_10, &x);

    /* Initialize rx pin */
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.odos  = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.podrv = ALD_GPIO_OUT_DRIVE_1;
    x.nodrv = ALD_GPIO_OUT_DRIVE_1;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_TTL;
    x.func  = ALD_GPIO_FUNC_3;
    ald_gpio_init(GPIOB, GPIO_PIN_11, &x);

    return;
}

/**
  * @brief  Send message complete.
  * @param  arg: Pointer to ald_uart_handle_t structure.
  * @retval None.
  */
void uart_send_complete(ald_uart_handle_t *arg)
{
    return;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to ald_uart_handle_t structure.
  * @retval None.
  */
void uart_recv_complete(ald_uart_handle_t *arg)
{
    return;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to ald_uart_handle_t structure.
  * @retval None.
  */
void uart_error(ald_uart_handle_t *arg)
{
    return;
}

/**
  * @brief  Polling method for sending 9-bit data.
  * @param  hperh: Pointer to a ald_uart_handle_t structure.
  * @param  buf: Starting address of the send data buffer.
  * @param  size: Send length.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_uart_send_9bit(ald_uart_handle_t *hperh, uint16_t *buf, uint16_t size)
{
    SET_BIT(hperh->perh->FCON, UART_FCON_TFRST_MSK);

    while (size-- > 0)
    {
        while (hperh->perh->STAT & (0x1 << 17));

        hperh->perh->TXBUF = (*buf++ & 0xFFF);
        hperh->tx_count++;
    }

    return ALD_OK;
}

/**
  * @brief  Polling method for receiving 9-bit data.
  * @param  hperh: Pointer to a ald_uart_handle_t structure.
  * @param  buf: Starting address of the receive data buffer.
  * @param  size: Receive length.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_uart_recv_9bit(ald_uart_handle_t *hperh, uint16_t *buf, uint16_t size)
{
    uint32_t stat = 0;

    SET_BIT(hperh->perh->RS485, UART_RS485_AADNEN_MSK);
    
    while (size-- > 0)
    {   
        /* Waiting for flag */
        while (1)
        {
            stat = hperh->perh->STAT;

            if (READ_BIT(stat, ALD_UART_STATUS_RFTH) != RESET)
                break;
        }
        
        *buf++ = (uint16_t)(hperh->perh->RXBUF & 0xFFF);
        hperh->rx_count++;
    }
   
    return ALD_OK;
}

/**
  * @brief  Enable the interrupt for receiving 9-bit data.
  * @param  hperh: Pointer to a ald_uart_handle_t structure.
  * @param  size: Receive length.
  * @retval None.
  */
void ald_uart_recv_9bit_by_it(ald_uart_handle_t *hperh, uint16_t size)
{
    hperh->rx_size  = size;
    hperh->rx_count = 0;
    hperh->err_code = ALD_UART_ERROR_NONE;
    
    SET_BIT(hperh->perh->RS485, UART_RS485_AADNEN_MSK);
    SET_BIT(hperh->perh->FCON, UART_FCON_RFRST_MSK);
    SET_BIT(hperh->perh->ICR, UART_ICR_RFTH_MSK);
    ald_uart_interrupt_config(hperh, ALD_UART_IT_RFTH, ENABLE);
}

/**
  * @brief  Interrupt handler for 9-bit UART data reception.
  * @param  hperh: Pointer to a ald_uart_handle_t structure.
  * @param  dst_buf: Starting address of the receive data buffer.
  * @retval None.
  */
void ald_uart_receive_9bit_handler(ald_uart_handle_t *hperh, uint16_t* dst_buf)
{
    /* Receive */
    if ((ald_uart_get_mask_flag_status(hperh, ALD_UART_IF_RFTH)) != RESET)
    {
        /* Get FIFO received data */
        while (READ_BITS(hperh->perh->FCON, UART_FCON_RXFL_MSK, UART_FCON_RXFL_POSS))
        {
            rx_buf[hperh->rx_count] = (uint16_t)(hperh->perh->RXBUF & 0xFFF);
            hperh->rx_count++;

            if (hperh->rx_count >= hperh->rx_size)
                break;
        }
        /* Clear irq flag */
        hperh->perh->ICR = ALD_UART_IF_RFTH;

        /* Call receive callback function */
        if (hperh->rx_count >= hperh->rx_size)
        {
            ald_uart_interrupt_config(hperh, ALD_UART_IT_RFTH, DISABLE);

            if (h_uart.rx_cplt_cbk)
                h_uart.rx_cplt_cbk(&h_uart);
        }
    }
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    uint8_t i = 0;
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HRC, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);  
    ald_mcu_irq_config(UART0_IRQn, 0, 3, ENABLE);

    /* clear ald_uart_handle_t structure */
    memset(&h_uart, 0x0, sizeof(h_uart));

    /* Initialize pin */
    uart_pin_init();

    /* Initialize tx_buf */
    for (i = 0; i < 8; ++i)
    {
        if (i % 3)
            tx_buf[i] = 0x1A0 + i;
        else
            tx_buf[i] = 0x0A1 + i;
    }
    /* Initialize rx_buf */
    memset(rx_buf, 0x00, sizeof(tx_buf)/sizeof(uint8_t));

    /* Initialize uart */
    h_uart.perh             = UART0;
    h_uart.init.baud        = 115200;
    h_uart.init.word_length = ALD_UART_WORD_LENGTH_8B;
    h_uart.init.stop_bits   = ALD_UART_STOP_BITS_2;
    h_uart.init.parity      = ALD_UART_PARITY_NONE;
    h_uart.init.mode        = ALD_UART_MODE_RS485;
    h_uart.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    h_uart.tx_cplt_cbk      = uart_send_complete;
    h_uart.rx_cplt_cbk      = uart_recv_complete;
    h_uart.error_cbk        = uart_error;
    ald_uart_init(&h_uart);

    /* Sending data using flag polling method */
    ald_uart_send_9bit(&h_uart, tx_buf, 8);
    ald_delay_ms(10);
    /* Receiving through query method */
    ald_uart_recv_9bit(&h_uart, rx_buf, 8);
    /* Receiving through interrupt method */
    ald_uart_recv_9bit_by_it(&h_uart, 8);

    while (1)
    {
        ald_uart_send_9bit(&h_uart, tx_buf, 8);
        ald_delay_ms(100);
    }
}

/**
  * @brief  Uart0 handler
  * @retval None
  */
void UART0_Handler(void)
{       
   ald_uart_receive_9bit_handler(&h_uart, rx_buf); 
}

/**
  * @}
  */
/**
  * @}
  */
